package com.openexchange.guard.server;

import java.util.ArrayList;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.bouncycastle.openpgp.PGPException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.database.Access;
import com.openexchange.guard.database.deletedKey.DeletedKey;
import com.openexchange.guard.database.deletedKey.DeletedKeysStorage;
import com.openexchange.guard.database.deletedKey.RestDbDeletedKeysStorage;
import com.openexchange.guard.encr.EncrLib;
import com.openexchange.guard.util.Core;


/**
 * Class for handling retrieving deleted keys
 * @author greg
 *
 */
public class KeyRetriever {

    private static Logger logger = LoggerFactory.getLogger(KeyRetriever.class);

    /**
     * Initial login for key retrieval.  Password verified.  Salt verified
     * Keys listed in response
     * @param request
     * @param response
     * @param cookie
     * @throws Exception
     */
    public void login (HttpServletRequest request, HttpServletResponse response, OxCookie cookie) throws Exception {
        JsonObject json = Core.getJSON(request);
        int userid= Core.getIntFromJson(json, "userid", true);
        int cid = Core.getIntFromJson(json, "cid", true);
        String email = Core.getStringFromJson(json, "username", true);
        String salt = Core.getStringFromJson(json, "salt", true);
        String password = Core.getStringFromJson(json, "password", true);
        JsonObject returnJson = new JsonObject();
        if (CheckBad.isBad("retr-" + email, Config.bad_password_count)) {
            logger.warn("Account lockout for key retrieval");
            sendBad(response, "lockout");
            return;
        }

        DeletedKeysStorage deletedKeysStorage = new RestDbDeletedKeysStorage();
        ArrayList<DeletedKey> keys = deletedKeysStorage.getAllExposedForEmail(email, userid, cid);
        if (keys.size() == 0) {
            sendBad(response, "not found");
            return;
        }
        if (!keys.get(0).verifyPassword(password)) {
            CheckBad.addBad("retr-" + email);
            logger.warn("Bad password attempt for key retrieval");
            sendBad(response, "Bad Password");
            return;
        }

        // Check salt matches at least on key
        boolean match = false;
        int i = 0;
        while ((i < keys.size()) && !match) {
            if (keys.get(i++).getSalt().equals(salt.trim())) {
                match=true;
            }
        }

        if (!match) {
            CheckBad.addBad("retr-" + email);
            sendBad(response, "bad salt");
            return;
        }

        // Ok, key password and salt verified, send data

        // Add the authorization data here
        logger.info("Sending retrieval key list for user " + email);
        JsonObject authdata = new JsonObject();
        Access acc = new Access();
        EncrLib encr = new EncrLib();
        String token = acc.getToken(cookie.OxReaderID);
        if (token == null) {
            token = acc.newToken(cookie.OxReaderID, userid, cid);
        }
        authdata.addProperty("encr_password", password);
        authdata.addProperty("user_id", userid);
        authdata.addProperty("cid", cid);
        String auth = encr.encryptAES(authdata.toString(), token);
        returnJson.addProperty("auth", auth);
        returnJson.add("keys", getKeyData(keys));
        Core.sendJson(response, returnJson);
        return;

    }

    /**
     * Retrieve a selected key by index
     * Must have auth for primary key
     * @param request
     * @param response
     * @param cookie
     * @throws Exception
     */
    public void getkeys (HttpServletRequest request, HttpServletResponse response, OxCookie cookie) throws Exception {
        JsonObject json = Core.getJSON(request);
        String email = Core.getStringFromJson(json, "username");
        String auth = Core.getStringParameter(request, "auth", true);
        if (CheckBad.isBad("retr-" + email, Config.bad_password_count))
        {
            Core.sendFail(response, "lockout");
            return;
        }
        cookie.JSESSIONID = cookie.OxReaderID;
        UserData userdata = new UserData(auth, cookie);
        if (userdata.userid<0){
            Core.sendFail(response, "Bad auth");
            CheckBad.addBad("retr-" + email);
            logger.warn("Bad password attempt for key retrieval");
            return;
        }
        int index = Core.getIntFromJson(json, "index", true);
        String password = Core.getStringFromJson(json, "password");
        if (password == null) {
            password = "";
        }
        DeletedKeysStorage deletedKeysStorage = new RestDbDeletedKeysStorage();
        ArrayList<DeletedKey> keys = deletedKeysStorage.getAllExposedForEmail(email, userdata.userid, userdata.cid);
        if (keys.size() == 0) {
            Core.sendFail(response, "not found");
            return;
        }
        if (index > keys.size()) {
            Core.sendFail(response, "out of range");
            return;
        }
        // Verify first password is correct.
        if (!keys.get(0).verifyPassword(userdata.encr_password)) {
            CheckBad.addBad("retr-" + email);
            Core.sendFail(response, "Bad Password");
            logger.warn("Bad password attempt for key retrieval");
            return;
        }
        String export;
        try {
            export = keys.get(index).export(password.equals("") ? userdata.encr_password : password);
        } catch (PGPException e) {
            Core.sendFail(response, "bad password");
            CheckBad.addBad("retr-" + email);
            return;
        }
        logger.info("Exported previously deleted key for " + email);
        JsonObject returnJson = new JsonObject();
        returnJson.add("keydata", getData(keys.get(index)));
        returnJson.addProperty("data", export);
        Core.sendJson(response, returnJson);
    }

    /**
     * Send a json response back with auth set to resp value
     * @param response
     * @param resp
     */
    private void sendBad (HttpServletResponse response, String resp) {
        JsonObject returnJson = new JsonObject();
        returnJson.addProperty("auth", resp);
        Core.sendJson(response, returnJson);
        return;
    }

    /**
     * Get key data from arraylist
     * @param keys
     * @return
     */
    private JsonArray getKeyData (ArrayList<DeletedKey> keys) {
        JsonArray keyArray = new JsonArray();
        for (int i = 0; i < keys.size(); i++) {
            JsonObject keyJson = getData (keys.get(i));
            keyJson.addProperty("index", i);
            keyArray.add(keyJson);
        }
        return (keyArray);
    }

    /**
     * Get the data from a deleted key
     * @param key
     * @return
     */
    private JsonObject getData (DeletedKey key) {
        JsonObject keyJson = new JsonObject();
        ArrayList<String> userids = key.getUserIds();
        JsonArray uids = new JsonArray();
        for (String uid : userids) {
            uids.add(new JsonPrimitive(uid));
        }
        keyJson.add("ids", uids);
        ArrayList<String> fingerprints = key.getKeyFingerprints();
        JsonArray fpArray = new JsonArray();
        for (String fp : fingerprints) {
            fpArray.add(new JsonPrimitive(fp));
        }
        keyJson.add("fingerprints", fpArray);
        return (keyJson);
    }

}
