/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.server;

import java.io.IOException;
import java.util.List;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.config.OxConfiguration;
import com.openexchange.guard.database.Access;
import com.openexchange.guard.database.DbCommand;
import com.openexchange.guard.database.DbQuery;
import com.openexchange.guard.database.GetPubKey;
import com.openexchange.guard.database.Keymanager;
import com.openexchange.guard.database.RecipKey;
import com.openexchange.guard.database.Sharding;
import com.openexchange.guard.database.ogKeyTable.OGKeyTableStorage;
import com.openexchange.guard.database.ogKeyTable.RestDbOGKeyTableStorage;
import com.openexchange.guard.encr.EncrLib;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.encr.RSAGen;
import com.openexchange.guard.exceptions.BadPasswordException;
import com.openexchange.guard.exceptions.BadRSAException;
import com.openexchange.guard.exceptions.GuardMissingParameter;
import com.openexchange.guard.logging.LogAction;
import com.openexchange.guard.logging.LogMarkers;
import com.openexchange.guard.mailcreator.Mail;
import com.openexchange.guard.ox.Api;
import com.openexchange.guard.ox.Languages;
import com.openexchange.guard.ox.Upgrader;
import com.openexchange.guard.pgp.PgpKeys;
import com.openexchange.guard.util.Core;
import com.openexchange.guard.validator.EmailValidator;


public class Auth {

	private static Logger logger = LoggerFactory.getLogger(Auth.class);


    public void createKeys(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception {
        JsonObject json = Core.getJSON(request);
    	Api ap = new Api(cookie, request);
        if (ap.verifyLogin() == false) {
        	logger.error("Create keys but not logged into UI");
            response.setStatus(550);
            return;
        }
        logger.debug("Starting to create master key");
        Access acc = new Access();
        int userid = Core.getIntFromJson(json, "user_id", true);
        int cid = Core.getIntFromJson(json, "cid", true);
        String language = Core.getStringFromJson(json, "language");
        String password = Core.getStringFromJson(json, "password", true);
        GuardKeys keys = acc.getKeys(userid, cid);
        if (keys == null) { // make sure already don'e exist
            String email = acc.getEmail(userid, cid);
            // Check we don't already have keys for that email
            GuardKeys prevKey = acc.getKeysFromEmail(email);
            if (prevKey != null) {
            	logger.error("Attempting to create keys that exist in other context");
            	Core.sendFail(response, "Existing");
            	return;
            }
            String name = email;
            if (json.get("name") != null) {
                name = json.get("name").getAsString();
            }
            Keymanager keyman = new Keymanager();
            keys = keyman.createMasterKeys(userid, cid, email, name, password, language, true, Keymanager.checkCreateRecovery(cookie, request),true); // Flag true for user created
            if (keys == null) {
                logger.debug("Failed to create master key");
                response.getWriter().write(" {\"result\":\"fail \"} ");
            } else {
            	if (json.has("email")) {
            		String recEmail = json.get("email").getAsString();
            		if (EmailValidator.validate(recEmail)) {
            			acc.storeQuestion(userid, cid, "e", recEmail);
            		}
            	}
            	LogAction.setSuccess(true);
            	logger.info("Created master key for user");
                logger.debug("Success creating master key");
                response.getWriter().write(" {\"result\":\"ok\"} ");
            }
        } else {  // Keys already exist, adding additional
            boolean adding = Core.getBooleanParameter(request, "add");
            if (adding) {
                String email = acc.getEmail(userid, cid);
                String name = email;
                if (json.get("name") != null) {
                    name = json.get("name").getAsString();
                }

                RSAGen gen = new RSAGen();
                GuardKeys newkey = gen.generateKeyPair(name, email, password);
                newkey.contextid = keys.contextid;
                newkey.userid = keys.userid;
                String recovery = Keymanager.createRecovery(newkey, password);
                //acc.addNewKey(newkey, keys, recovery);
                newkey.recovery = recovery;
                acc.addNewKey(newkey);
                LogAction.setSuccess(true);
                logger.info("Added additional key for user");
                Core.sendOK(response, " {\"result\":\"ok\"} ");
            } else {
                LogAction.setFail();
                logger.info("Fail creating key, already exists and add flag not set");
                Core.sendFail(response, "Existing");
                return;
            }

        }
    }

    /**
     * Change password
     * @param request
     * @param response
     * @param cookie
     * @return
     * @throws Exception
     */
    public String changepass(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception {
        JsonObject json = Core.getJSON(request);
        if (CheckBad.isBad(cookie.JSESSIONID, Config.bad_password_count)) {
        	logger.debug("Lockout for bad attempts");
            response.getWriter().write("{\"auth\":\"Lockout\"}");
            return (null);
        }
        String newpass = Core.getStringFromJson(json, "newpass", true).trim();
        if (newpass.length() < Config.min_password_length) {
        	response.getWriter().write("{\"auth\":\"Bad new password\",\"minlength\":" + Config.min_password_length + "}");
            return (null);
        }
        int cid = Core.getIntFromJson(json, "cid", true);
        String question = "";
        if (json.has("question")) { // Sent with new question and answer?
            question = json.get("question").getAsString();
        }
        String answer = "";
        if (json.has("answer")) {
            answer = json.get("answer").getAsString();
        }
        if (json.has("email")) {
        	question = "e";
        	answer = json.get("email").getAsString();
        }
        // if ox member, verify actually logged in to ui
        if (cid > 0) {
            Api ap = new Api(cookie, request);
            if (ap.verifyLogin() == false) {
                response.setStatus(550);
                return (null);
            }
        }

        // OK, logged in or guest, so we can change
        response.setContentType("application/json");
        response.addHeader("Access-Control-Allow-Origin", "*");
        logger.debug("change password request");
        int userid = Core.getIntFromJson(json, "user_id", true);

        com.openexchange.guard.database.Access acc = new com.openexchange.guard.database.Access();
        GuardKeys keys = acc.getKeys(userid, cid);
        if (keys == null) {
        	response.getWriter().write("{\"auth\":\"Key not found\"}");
        	return (null);
        }
        String oldpass = "";
        oldpass = Core.getStringFromJson(json, "oldpass", true).trim();
        PgpKeys keygen = new PgpKeys();
        keys = keygen.changePassword(oldpass, newpass, keys);
        if (keys == null) {
            LogAction.setFail();
            logger.info("Bad password for change");
            response.getWriter().write("{\"auth\":\"Bad password\"}");
            CheckBad.addBad(cookie.JSESSIONID);
            return (null);
        } else {
            String recovery = "";
            boolean recoveryAvail = false;
            if (Keymanager.checkCreateRecovery(cookie, request)) {
                recoveryAvail = true;
                recovery = Keymanager.createRecovery(keys, newpass);
            }
            acc.updateKeys(userid, cid, keys, recovery, false);
            if (!answer.equals("")) {// If sent with question and answer, then store
                if (question.equals("")) {
                    question = "default";
                }
                acc.storeQuestion(userid, cid, question, answer);
            }
            // String sessionID = json.get("sessionID").getAsString();
            String token = acc.getToken(cookie.JSESSIONID);
            if (token == null) {
                token = acc.newToken(cookie.JSESSIONID, userid, cid);
            }
            json.add("encr_password", new JsonPrimitive(newpass));
            EncrLib encr = new EncrLib();
            JsonObject returndata = new JsonObject();
            returndata.addProperty("auth", encr.encryptAES(json.toString(), token));
            returndata.addProperty("recovery", recoveryAvail);
            Core.sendOK(response, returndata.toString());
            LogAction.setSuccess();
            logger.info("OK Password Change");
            return (returndata.toString());
        }

    }

    /**
     * Update secondary email address after verifying OxGuard password
     * @param request
     * @param response
     * @param cookie
     * @throws Exception
     */
    public void changeSecondaryEmail (HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception {
    	JsonObject json = Core.getJSON(request);
    	int cid = Core.getIntFromJson(json, "cid", true);
    	int userid = Core.getIntFromJson(json, "userid", true);
    	if (CheckBad.isBad(Integer.toString(userid) + "-" + Integer.toString(cid), Config.bad_password_count)) {
    		Core.sendFail(response, "Lockout");
    		return;
    	}
    	Api ap = new Api(cookie, request);
        if (ap.verifyLogin() == false) {// verify logged into UI
            Core.sendFail(response, "Not logged in");
            return;
        }
        String password = Core.getStringFromJson(json, "password", true);
        String newemail = Core.getStringFromJson(json, "email", true);
        Access acc = new Access();
        GuardKeys keys = acc.getKeys(userid, cid);
        if (keys.verifyPassword(password)) {
        	if (updateEmail (userid, cid, newemail)) {
        	    LogAction.setSuccess();
        		Core.sendOK(response, "OK");
        	} else {
        	    LogAction.setFail();
        	    logger.info("Problem updating secondary email");
        	    Core.sendFail(response, "Fail");
        	}
        } else {
        	CheckBad.addBad(Integer.toString(userid) + "-" + Integer.toString(cid));
        	LogAction.setFail();
        	logger.info("Bad password for changing secondary email");
        	Core.sendFail(response, "Bad password");
        	return;
        }

    }

    public void getSecondaryEmail (HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception {
    	Api ap = new Api(cookie, request);
        if (ap.verifyLogin() == false) {// verify logged into UI
            Core.sendFail(response, "Not logged in");
            return;
        }
        int cid = Core.getIntParameter(request, "cid", true);
        int userid = Core.getIntParameter(request, "userid", true);
        LogAction.setUser(userid, cid);
       	String command = "SELECT answer, email FROM og_KeyTable WHERE id = ? AND CID = ?";
    	DbCommand dcom = new DbCommand(command);
    	dcom.addVariables(userid);
    	dcom.addVariables(cid);
    	DbQuery db = new DbQuery();
    	db.read(dcom, userid, cid);
    	if (db.next()) {
    		String encr_answer = db.rs.getString("answer");
    		EncrLib encr = new EncrLib();
    		String secondEmail = encr.decryptAES(encr_answer, Config.rpass, "e", Config.AESKeyLen, false);
    		if (secondEmail != null) {
    			db.close();
    			Core.sendOK(response, secondEmail);
    			return;
    		}
    	}
    	db.close();
    	LogAction.setSuccess();
    	Core.sendOK(response, "");
    }

    private boolean updateEmail (int userid, int cid, String email) {
    	Access acc = new Access();
    	return(acc.storeQuestion(userid, cid, "e", email));
    }

    public void resetPass(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception {
    	int cid = Core.getIntParameter(request, "cid", true);
        int userid = Core.getIntParameter(request, "userid", true);
        LogAction.setUser(userid, cid);
        Api ap = new Api(cookie, request);
        if (ap.verifyLogin() != false) {
            resetPass (userid, cid, request, response, cookie, true);
        } else {
            response.setStatus(550);
            return;
        }
    }

    /*
     * Reset password.  Used from command line tools only
     */
    public void resetPass (String email) {
    	Access acc = new Access();
			try {
				GuardKeys keys = acc.getKeysFromEmail(email);
				if (keys == null) {
					System.out.println("NO ACCOUNT FOUND");
					return;
				}
				if (keys.contextid < 0) { // Guest
					Keymanager keyman = new Keymanager();
					String newpass = keyman.resetPassword(email);
					if (newpass.equals("NR")) {
						System.out.println("No recovery available");
						return;
					}
	    			Sender sender = new Sender();
	                com.openexchange.guard.mailcreator.Mail mf = new com.openexchange.guard.mailcreator.Mail();
	                int templid = 0;
	                try {
	                    templid = OxConfiguration.getTemplateId(keys.contextid, keys.userid);
	                } catch (Exception e) {
	                    logger.error("problem getting template id for reset password email");
	                }
	                if (!sender.send(mf.getResetEmail(email, Mail.getFromAddress(email, email, keys.userid, keys.contextid), newpass, Config.default_lang, templid), null)) {
	                	System.out.println("Failed to send");
	                } else {
	                	System.out.println("Reset sent");
	                }
				} else {
                    resetPass (keys.userid, keys.contextid, null, null, null, false);
                }
			} catch (Exception e) {
			    LogAction.setFail();
				logger.error("Error resetting password for " + email, e);
			}

    }

    // Remove PIN, for command line use only
    public static void removePin (String email) {
    	Access acc = new Access();
		try {
			GuardKeys keys = acc.getKeysFromEmail(email);
			if (keys == null) {
				System.out.println("NO ACCOUNT FOUND");
				return;
			}
			if (keys.contextid > 0) { // OX email
				System.out.println("Not a guest account");
				return;
			}
		    acc.storeQuestion(keys.userid, keys.contextid, null, null);
		    System.out.println("done");
		} catch (Exception ex) {
			System.out.println("Error resetting PIN");
			System.out.println(ex);
		}
    }

    /**
     * Delete password recovery for the user
     * @param request
     * @param response
     * @param cookie
     * @throws Exception
     */
    public void deleteRecovery (HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception {

        JsonObject json = Core.getJSON(request);
        Api ap = new Api(cookie, request);
        // verify login
        if (!ap.verifyLogin()) {
            Core.sendFail(response, "Must be logged in");
            return;
        }
        String password = Core.getStringFromJson(json, "password", true);
        int userid = Core.getIntFromJson(json, "userid", true);
        int cid = Core.getIntFromJson(json, "cid", true);

        Access acc = new Access();
        GuardKeys key = acc.getKeys(userid, cid);

        // verify password
        if (!key.verifyPassword(password)) {
            Core.sendFail(response, "Bad password");
            return;
        }

        OGKeyTableStorage ogKeyTableStorage = new RestDbOGKeyTableStorage();
        ogKeyTableStorage.deleteRecovery(key);
        Core.sendOK(response, "OK");
    }

    public void resetPass(int userid, int cid, HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie, boolean web) throws Exception {
        int templid = 0;
        try {
            templid = OxConfiguration.getTemplateId(cid, userid);
        } catch (Exception e) {
            logger.error("problem getting template id for reset password email");
        }
        String lang = request == null ? Config.default_lang : request.getParameter("language");
    	String command = "SELECT answer, email FROM og_KeyTable WHERE id = ? AND CID = ?";
    	DbCommand dcom = new DbCommand(command);
    	dcom.addVariables(userid);
    	dcom.addVariables(cid);
    	DbQuery db = new DbQuery();
    	db.read(dcom, userid, cid);
    	if (db.next()) {
    		String encr_answer = db.rs.getString("answer");
    		String userEmail = db.rs.getString("email");
    		EncrLib encr = new EncrLib();
    		boolean secondary = true;
    		// Mailing to secondary email address if web.  If from command line, then email to primary
    		String secondEmail = web ? encr.decryptAES(encr_answer, Config.rpass, "e", Config.AESKeyLen, false) : userEmail;
    		if ((secondEmail == null) || secondEmail.equals("")) {
    			secondEmail = userEmail;
    			secondary = false;
    		}
    		if (EmailValidator.validate(secondEmail)) {
    			Keymanager keyman = new Keymanager();
    			String newpass = keyman.resetPassword(userEmail);
    			if (newpass == null) {
    				if (web) {
    				    LogAction.setFail();
                    	Core.sendOK(response, "FailChange");
                    } else {
                        System.out.println("Failed to change");
                    }
    				db.close();
    				return;
    			}
    			if (newpass.equals("NR")) {
    				if (web) {
    				    LogAction.setFail();
    				    logger.info("Unable to recover password due to no recovery");
                    	Core.sendOK(response, "NoRecovery");
                    } else {
                        System.out.println("No recovery available");
                    }
    				db.close();
    				return;
    			}
    			Sender sender = new Sender();
                com.openexchange.guard.mailcreator.Mail mf = new com.openexchange.guard.mailcreator.Mail();
                String from = userEmail;
                if (!sender.send(mf.getResetEmail(secondEmail, Mail.getFromAddress(from, from, userid, cid) , newpass, lang, templid), null)) {
                    if (web) {
                        LogAction.setFail();
                        logger.error("Failed to send email reset");
                    	Core.sendOK(response, "FailNotify");
                    } else {
                        System.out.println("Failed to send nofication");
                    }
                    db.close();
                    return;
                } else {
                    if (web) {
                        LogAction.setSuccess();
                    	if (secondary) {
                    		Core.sendOK(response, "ok");
                    	} else {
                    		Core.sendOK(response, "primary");
                    	}
                    } else {
                        System.out.println("OK");
                    }
                }
    		} else { // Bad secondary email
    			if (web) {
                	Core.sendOK(response, "NoSecondary");
                } else {
                    System.out.println("No Secondary Email");
                }
                db.close();
                return;
    		}
    	}
    	db.close();
    }
    /**
     * Reset password from within OX UI using OX login password --- Depreciated
     *
     * @param request
     * @param response
     * @param cookie
     * @return
     * @throws Exception
     */
    public String changepassFromOxRecovery(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception {
        JsonObject json = Core.getJSON(request);
        int cid = Core.getIntFromJson(json, "cid", true);
        String password = Core.getStringFromJson(json, "oxpassword", true);
        int userid = Core.getIntFromJson(json, "user_id", true);
        String user = Core.getStringFromJson(json, "user", true);
        com.openexchange.guard.database.Access acc = new com.openexchange.guard.database.Access();
        GuardKeys keys = acc.getKeys(userid, cid);
        // if ox member, verify actually logged in to ui
        if (cid > 0) {
            Api ap = new Api(cookie, request);
            if (ap.verifyLogin() == false) {
                response.setStatus(550);
                return (null);
            }
        }
        Api ap2 = new Api(user, password);
        if (!ap2.verifyLogin()) {
            response.setStatus(550);
            return (null);
        }

        // OK, logged in , so we can change

        Keymanager keyman = new Keymanager();
        String newpass = Core.getStringFromJson(json, "newpass", true);
        String resetpassword = keyman.resetPassword(keys.email, newpass);
        response.setContentType("application/json");
        response.addHeader("Access-Control-Allow-Origin", "*");
        if (resetpassword == null) {
            LogAction.setFail();
            logger.info("Bad password for change");
            response.getWriter().write("{\"auth\":\"Bad password\"}");
            return (null);
        } else {
            String sessionID = json.get("sessionID").getAsString();
            String token = acc.newToken(cookie.JSESSIONID, userid, cid);
            json.add("encr_password", new JsonPrimitive(newpass));
            EncrLib encr = new EncrLib();
            String returndata = " {\"auth\":\"" + encr.encryptAES(json.toString(), token) + "\"} ";
            logger.debug(returndata);
            response.getWriter().write(returndata);
            LogAction.setSuccess();
            logger.info("OK Password Change");
            return (returndata);
        }

    }

    /**
     * Authorize login from guest account
     *
     * @param request
     * @param response
     * @throws GuardMissingParameter
     * @throws BadRSAException
     */
    public void guestlogin(HttpServletRequest request, HttpServletResponse response) throws GuardMissingParameter, BadRSAException {
    	JsonObject login_json = Core.getJSON(request);
        com.openexchange.guard.database.Access acc = new com.openexchange.guard.database.Access();
        response.setContentType("application/json");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        response.addHeader("Connection", "close");
        String sessionID = EncrLib.getUUID();
        String cookieID = EncrLib.getUUID();
        Cookie cookie = new Cookie("OxReaderID", cookieID);
        cookie.setHttpOnly(true);
        cookie.setPath("/");
        response.addCookie(cookie);
        String username = Core.getStringFromJson(login_json, "username", true);
        String pin = login_json.get("pin") == null ? null : login_json.get("pin").getAsString();
        try {
            if (CheckBad.isBad(username, Config.bad_password_count)) {
            	response.getWriter().write("{\"auth\":\"lockout\", \"time:\": " + Config.bad_minute_lock + "}");
            	LogAction.setFail();
            	logger.info("Guest auth lockout for " + username);
                return;
            }
            String password = Core.getStringFromJson(login_json, "password", true);
            GuardKeys keys = acc.getKeysFromEmail(username);
            if (keys == null) {
                response.getWriter().write("{\"auth\":\"No Key\"}");
                LogAction.setFail();
                logger.info("Guest auth No Key for " + username);
                return;
            }


            String token = acc.newToken(cookieID, keys.userid, keys.contextid);
            if (token == null) {
                response.getWriter().write("{\"auth\":\"Unable to get token\"}");
                LogAction.setFail();
                logger.error("Guest auth Unable to get database token for " + username);
                return;
            }
            // This is the json we're going to encrypt and include in the auth
            JsonObject json = new JsonObject();
            json.addProperty("user_id", keys.userid);
            json.addProperty("encr_password", password);
            json.addProperty("sessionID", sessionID);
            json.addProperty("cid", keys.contextid);
            EncrLib encr = new EncrLib();
            String returndata = "Bad Password";

            if (!keys.verifyPassword(password)) {
                CheckBad.addBad(username);
                LogAction.setFail();
                logger.info("Bad password for " + username);
            } else
            {
                LogAction.setSuccess();
                logger.info("Guest auth good for " + username);
                // Check if PIN

                if ((keys.lastup == null) && (keys.question == null ? false : keys.question.equals("PIN"))) {
                	if (pin != null) {
                		if (pin.equals(keys.answer)) {
                			returndata = encr.encryptAES(json.toString(), token);
                		} else {
                			returndata = "Bad";
                			CheckBad.addBad(username);
                		}
                	} else {
                		logger.info("New Guest login, pin required");
                		returndata = "PIN";
                	}
                } else {
                	returndata = encr.encryptAES(json.toString(), token);
                }
            }

            // The json returned in plain text.  no password here (encrypted within auth)  Include settings
            JsonObject sendjson = new JsonObject();
            sendjson.addProperty("auth", returndata);
            sendjson.addProperty("userid", keys.userid);
            sendjson.addProperty("sessionID", sessionID);
            sendjson.addProperty("cid", keys.contextid);
            sendjson.addProperty("lastMod", keys.lastup == null ? "null" : keys.lastup.toString());
            try {
            	Settings settings = new Settings();
            	sendjson.add("settings", getSettings(keys.settings.toString(), settings, keys.userid, keys.contextid));
            } catch (Exception ex) {
            	logger.error("problem getting guest settings " , ex);
            }
            response.getWriter().write(sendjson.toString());

        } catch (Exception e) {
            LogAction.setFail();
            logger.error("Error during guest login  for user " + username, e);
        }
    }

    public void logout(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception {
        int userid = Core.getIntParameter(request, "userid");
        int cid = Core.getIntParameter(request, "cid");
        LogAction.setUser(userid, cid);
        com.openexchange.guard.database.Access acc = new com.openexchange.guard.database.Access();
        acc.deleteToken(cookie.JSESSIONID);
        acc.deleteToken(cookie.OxReaderID);
        com.openexchange.guard.util.Core.sendOK(response, "OK");
    }

    public void login(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws IOException {
        try {

            JsonObject json = Core.getJSON(request);
            Settings settings = getPermissions(cookie, request);
            login(json, request, response, cookie, settings);
        } catch (BadRSAException e) {
            LogAction.setFail();
            logger.error("Unable to decode RSA password");
            JsonObject ret = new JsonObject();
            ret.addProperty("auth", "Bad RSA");
            Core.sendJson(response, ret);
            return;
        } catch (Exception ex) {
            LogAction.setFail();
            logger.error("Error reading login JSON information", ex);
            JsonObject ret = new JsonObject();
            ret.addProperty("auth", "BAD FORMAT");
            Core.sendJson(response, ret);
            return;
        }
    }

    public void login(JsonObject json, HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie, Settings settings) throws Exception {
        long start = System.currentTimeMillis();
        com.openexchange.guard.database.Access acc = new com.openexchange.guard.database.Access();
        response.setContentType("application/json");
        // response.setContentType("text/html");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        response.addHeader("Connection", "close");
        String lang = Core.getStringFromJson(json, "lang");
        JsonObject returnJson = new JsonObject ();
        returnJson.add("lang", Languages.getJsonAvail(lang));
        returnJson.addProperty("server", Config.serverVersion);
        // Get public RSA keys for password encryption
        String pubkey = Core.getPublicKey();
        if (pubkey != null) {
        	returnJson.addProperty("pubKey", pubkey);
        }
        int cid = 0;
        try {

            int userid = Core.getIntFromJson(json, "user_id", true);
            cid = Core.getIntFromJson(json, "cid", true);
            MDC.put("ID", Integer.toString(userid));
            MDC.put("CID", Integer.toString(cid));
            String email = json.get("email") == null ? null : json.get("email").getAsString();
            // Check lockout of bad attempts.  If excessive, don't give encrypted auth key
            if (CheckBad.isBad(Integer.toString(userid) + "-" + Integer.toString(cid), Config.bad_password_count)) {
            	returnJson.addProperty ("auth", "Lockout");
            	returnJson.add ("settings", getSettings(null, settings, userid, cid));
                response.getWriter().write(returnJson.toString());
                LogAction.setFail();
                logger.info("Lockout auth due to bad attempts");
                return;
            }
            if (cid == 0) { // Occasionally, the UI doesn't load the CID.  Get from backend
            	logger.debug("0 cid for userid " + userid);
            	Api ap = new Api(cookie, request);
            	cid = ap.getContextId(); // Returns same 0 if fails
            	json.remove("cid"); // Replace the cid in json
            	json.addProperty("cid", cid);
            	logger.debug("Retrieved context_id of " + cid);
            }
            returnJson.addProperty("cid", cid);
            GuardKeys keys = null;
            String keyid = Core.getStringParameter(request, "keyid");
            if (keyid != null) {
                try {
                    Long kid = Long.parseLong(keyid);
                    keys = acc.getKeys(userid, cid, kid, false);
                } catch (Exception ex) {
                    logger.error("Problem with login for Key ID " + keyid, ex);
                }
            } else {
                keys = acc.getKeys(userid, cid);
            }

            // Nothing found.  Previously a non-ox user and new to us?
            if (keys == null) {
            	if (email != null) {
	            	RecipKey r = GetPubKey.checkMaster(email);  // check to see if we have a public non-ox key
	            	if (r != null) {
	            		if (r.cid < 0) {
	            			try {
	            				Upgrader.upgrade(r.userid, r.cid, userid, cid); // try to upgrade the account
	            				keys = acc.getKeys(userid, cid); // try again
	            			} catch (Exception ex) {
	            				logger.error(LogMarkers.NOTIFY_MINOR, "Unable to upgrade email " + email);
	            			}

	            		}
	            	}
            	}
            }
            if (keys == null) {
                LogAction.setSuccess();
            	returnJson.addProperty ("auth", "No Key");
            	returnJson.add ("settings", getSettings(null, settings, userid, cid));
                response.getWriter().write(returnJson.toString());
                logger.debug("Auth sent with No Key for cid = " + cid + " userid = " + userid);
                return;
            }
        	returnJson.add ("settings", getSettings(keys.settings.toString(), settings, userid, cid));
        	returnJson.add("recoveryAvail", new JsonPrimitive(keys.recoveryAvail));
            if (keys.passwordNeeded == true) {
            	returnJson.addProperty ("auth", "Password Needed");
                response.getWriter().write(returnJson.toString());
                LogAction.setSuccess();
                logger.debug("Auth sent with Password Needed for cid = " + cid + " userid = " + userid);
                return;
            }

            String token = acc.getToken(cookie.JSESSIONID);
            if (token == null) {
                token = acc.newToken(cookie.JSESSIONID, userid, cid);
            }
            if (token == null) {
            	returnJson.addProperty ("auth", "Unable to get token");
                response.getWriter().write(returnJson.toString());
                LogAction.setFail();
                logger.error("Auth unable to get token from database for cid = " + cid + " userid = " + userid);
                return;
            }
            EncrLib encr = new EncrLib();
            try {
                String oxpassword = json.get("ox_password") == null ? "" : json.get("ox_password").getAsString();
                if (oxpassword.equals("null")) {
                    oxpassword = "";
                }
                String encrpass = Core.getStringFromJson(json, "encr_password");
                if (encrpass == null) {
                    encrpass = "";
                }
                String returndata = "Bad Password";

                    if (encrpass.length() > 1) { // If provided with encryption password, check
                        if (verifyPassword(keys, userid, cid, encrpass)) {
                            returndata = encr.encryptAES(json.toString(), token);
                            CheckBad.removeBad(Integer.toString(userid) + "-" + Integer.toString(cid)); // remove bad session listing count
                            logger.debug("Auth with proper password for cid = " + cid + " userid = " + userid);
                            LogAction.setSuccess();
                        } else {
                        	CheckBad.addBad(Integer.toString(userid) + "-" + Integer.toString(cid));  // Increment fail password counter
                        	LogAction.setSuccess();
                        	logger.debug("Auth, OG user, settings only for cid = " + cid + " userid = " + userid);
                        }
                    } else { // If no encr password, try the oxpassword
                        if (oxpassword.length() > 1) {
                        	try {
                                LogAction.setSuccess();
	                            if (verifyPassword(keys, userid, cid, oxpassword)) {
	                                json.remove("encr_password");
	                                json.addProperty("encr_password", oxpassword); // The encryption password is our ox password
	                                returndata = encr.encryptAES(json.toString(), token);
	                                logger.debug("Auth with proper password for cid = " + cid + " userid = " + userid);
	                                // returndata = encr.encryptAES(json.toString(), token, cookie.JSESSIONID, Config.AESKeyLen);
	                            } else {
	                            	CheckBad.addBad(Integer.toString(userid) + "-" + Integer.toString(cid));  // Increment fail password counter
	                            	logger.debug("Auth, OG user, settings only for cid = " + cid + " userid = " + userid);
	                            }
                        	} catch (BadPasswordException bp) {
                        		CheckBad.addBad(Integer.toString(userid) + "-" + Integer.toString(cid));  // Increment fail password counter
                            	logger.debug("Auth, OG user, settings only for cid = " + cid + " userid = " + userid);
                        	}
                        }
                    }

                if (json.has("extrapass")) { // File does extra login to encrypt passwords prior to creating links
                	String extrapass = json.get("extrapass").getAsString();
                	if (!extrapass.equals("")) {
                    	String epass = encr.encryptAES(extrapass, cookie.JSESSIONID, "epass", Config.AESKeyLen);
                    	returnJson.addProperty("encrExtra", epass);
                	}
                }
                returnJson.addProperty("auth", returndata);
                response.getWriter().write(returnJson.toString());
                LogAction.setSuccess();
                logger.info("Authorization at " + start);
            } catch (Exception e) {
                LogAction.setFail();
                logger.error("Error during login", e);
                returnJson.addProperty("auth", "BAD FORMAT");
                response.getWriter().write(returnJson.toString());
                return;
            }

        } catch (Exception e) {
            if (e.getMessage().contains("doesn't exist")) {
                if (cid > 0) {
                    Sharding.checkTables(cid, true);
                    logger.info("Created OG tables for cid " + cid);
                }
                returnJson.addProperty("auth", "No Key");
                returnJson.add ("settings", getSettings(null, settings, 0, 0));
                LogAction.setSuccess();
                response.getWriter().write(returnJson.toString());
            } else {
                response.getWriter().write("{\"auth\":\"BAD FORMAT\", \"lang\": " + Languages.getJsonAvail(Languages.default_code) + "} ");
                LogAction.setSuccess(false);
                logger.error("Error during login", e);
            }
        }

    }

    /**
     * First, verify the current password.  If fails, then check if one of the other keys
     * @param key
     * @param userid
     * @param cid
     * @param password
     * @return
     * @throws Exception
     */
    private boolean verifyPassword (GuardKeys key, int userid, int cid, String password) throws Exception {
        if (key.verifyPassword(password)) return (true);
        OGKeyTableStorage ogKeyTableStorage = new RestDbOGKeyTableStorage();
        List<GuardKeys> userKeys = ogKeyTableStorage.getKeysForUser(userid, cid);
        if (userKeys.size() > 1) {
            for (GuardKeys userKey : userKeys) {
                if (!userKey.current) {  // Don't recheck the current key
                    if (userKey.verifyPassword(password)) return (true);
                }
            }
        }
        return (false);
    }

    /**
     * Create a json of settings, include the individuals settings, plus global
     *
     * @param settings
     * @return
     */
    private JsonObject getSettings(String settings, Settings setting, int id, int cid) {
        StringBuilder allSettings = new StringBuilder();
        if (settings == null) {
            settings = "";
        }

        // If settings string loaded with key, convert to JSON
        if (!settings.startsWith("{")) {
            allSettings.append("{");
        }
        if (settings != null) {
            allSettings.append(settings);
        }
        if (!settings.endsWith("}")) {
            allSettings.append("}");
        }

        // Convert to Json, then check if global defaults present and add. Individual settings override
        Gson gson = new GsonBuilder().create();
        JsonObject json = gson.fromJson(allSettings.toString(), JsonObject.class);
        if (!json.has("Private")) { // If not defined at user level database, use global settings
        	if (setting.privatepass == false) { // If config cascade value set as false, then use
        		json.add("ChangePrivate", new JsonPrimitive("false"));
        	} else {
                json.add("ChangePrivate", new JsonPrimitive(Config.ok_private ? "true" : "false"));
            }
        }

        if (!json.has("extrapass")) { // If not defined at user level database, use global settings
        	json.add("extrapass", new JsonPrimitive(setting.extrapass ? "true" : "false"));

        }

        if (!json.has("deleteOnRevoke")) {
        	json.add("deleteOnRevoke", new JsonPrimitive(setting.deleteKeyOnRevoke ? "true" : "false"));
        }

        if (!json.has("oxguard")) {
            json.add("oxguard", new JsonPrimitive(check(setting, Settings.permissions.MAIL)));
        }

        // If either the global setting is set for no delete keys, or the config cascade, then use
        json.add("noDeletePrivate", new JsonPrimitive(setting.noDeletePrivate));
        json.add("noDeleteRecovery", new JsonPrimitive(setting.noDeleteRecovery));

        json.add("noRecovery", new JsonPrimitive(setting.norecovery));

        if (Config.showStatus == false) {
        	json.addProperty("status", false);
        }
        json.addProperty("min_password_length", Config.min_password_length);
        json.addProperty("password_length", Config.newPassLength);

        json.addProperty("secureReply", OxConfiguration.secureReply(id, cid));

        return (json);
    }

    private String toJson(String name, String value) {
        return ("\"" + name + "\":\"" + value + "\"");
    }

    public static Settings getPermissions(com.openexchange.guard.server.OxCookie cookie, HttpServletRequest request) {
    	Settings settings = new Settings();
        settings.perm = Settings.permissions.NONE;
        Api ap = new Api(cookie, request);
        String dat = ap.getCapabilities().toLowerCase();
        if (dat == null) {
        	logger.error("Unable to get permissions from OX backend");
        	return (settings);
        }
        if (dat.contains("guard")) {
            settings.perm = Settings.permissions.GUARD;
        }
        if (dat.contains("guard:mail") || dat.contains("guard-mail")) {
            settings.perm = Settings.permissions.MAIL;
        }
        if (dat.contains("guard:drive") || dat.contains("guard-drive")) {
            if (settings.perm == Settings.permissions.MAIL) {
                settings.perm = Settings.permissions.BOTH;
            } else {
                settings.perm = Settings.permissions.FILE;
            }
        }
        if (dat.contains("guard-noextra")) {
            settings.extrapass = false;
        }
        if (dat.contains("guard-noprivate")) {
            settings.privatepass = false;
        }
        if (dat.contains("guard-nodeleterecovery")) {
            settings.noDeleteRecovery = true;
        } else {
            settings.noDeleteRecovery = Config.no_delete_recovery;
        }

        if (dat.contains("guard-nodeleteonrevoke")) {
            settings.deleteKeyOnRevoke = false;
        }
        if (dat.contains("guard-norecovery")) {
            settings.norecovery = true;
        } else {
            settings.norecovery= Config.no_recovery;
        }

        if (dat.contains("guard-nodeleteprivate")) {
            settings.noDeletePrivate = true;
        } else {
            settings.noDeletePrivate = Config.no_delete_keys;
        }
        return (settings);
    }

    public static boolean checkPermission(com.openexchange.guard.server.OxCookie cookie, HttpServletRequest request, Settings.permissions req) {
        Settings settings = getPermissions(cookie, request);
        return (check(settings, req));
    }

    public static boolean checkMinimum (Settings current) {
        if ((current.perm == Settings.permissions.GUARD) || (current.perm == Settings.permissions.FILE)|| (current.perm == Settings.permissions.BOTH)) {
            return(true);
        }
        return(false);
    }

    public static boolean check(Settings current, Settings.permissions req) {
        if (req == current.perm) {
            return (true);
        }
        if (current.perm == Settings.permissions.BOTH) {
            return (true);
        }
        return (false);
    }
}
