/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgp;

import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.io.IOUtils;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPUserAttributeSubpacketVector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.amazonaws.services.cloudfront_2012_03_15.model.InvalidArgumentException;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.openexchange.guard.database.Access;
import com.openexchange.guard.database.Keymanager;
import com.openexchange.guard.database.ogKeyTable.OGKeyTableStorage;
import com.openexchange.guard.database.ogKeyTable.RestDbOGKeyTableStorage;
import com.openexchange.guard.database.pgpKeys.PGPKeys;
import com.openexchange.guard.database.pgpKeys.PGPKeysStorage;
import com.openexchange.guard.database.pgpKeys.RestDbPGPKeysStorage;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.exceptions.BadPasswordException;
import com.openexchange.guard.logging.LogAction;
import com.openexchange.guard.ox.Api;
import com.openexchange.guard.server.Auth;
import com.openexchange.guard.server.OxCookie;
import com.openexchange.guard.server.Settings;
import com.openexchange.guard.server.UserData;
import com.openexchange.guard.util.Core;


public class PGPPrivateKeyHandler {
    private static Logger logger = LoggerFactory.getLogger(PGPPrivateKeyHandler.class);

    public void incomingPGPForm(HttpServletRequest request, HttpServletResponse response, OxCookie cookie) throws Exception {
        FileItemFactory fif = new DiskFileItemFactory();
        ServletFileUpload sfu = new ServletFileUpload(fif);
        int userid = Core.getIntParameter(request, "userid", true);
        int cid = Core.getIntParameter(request, "cid", true);
        LogAction.setUser(userid, cid);
        List files = sfu.parseRequest(request);
        String sessionId = Core.getStringParameter(request, "session", true);
        String ua = request.getHeader("User-Agent");
        Iterator iterator = files.iterator(); // Iterate through attachments
        String privdata = "";
        String pubdata = "";
        String pgppassword = "";
        String password = null;

        //Check authentication
        Api ap = new Api(cookie, request);
        if (!ap.verifyLogin()) {
            LogAction.setFail();
            logger.info("User not logged into UI");
            Core.sendFail(response, "Must be logged in to UI");
            return;
        }
        while (iterator.hasNext()) {
            FileItem fi = (FileItem) iterator.next();
            String name = fi.getFieldName();
            if (name.equals("pgp")) {
                pgppassword = fi.getString("UTF-8");
                continue;
            }
            if (name.equals("password")) {
                password = fi.getString("UTF-8");
                if (password.equals("undefined")) {
                    password = null;
                }
                continue;
            }
            name = fi.getName();
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            InputStream input = fi.getInputStream();
            IOUtils.copy(input, out); // Copy stream to byte array
            String data = "\r\n" + new String(out.toByteArray()) + "\r\n";

            int start = data.indexOf("-----BEGIN PGP PRIV");
            if (start >= 0) {
                int end = data.indexOf("-----END PGP PRIVATE KEY BLOCK", start);
                end = data.indexOf("\r", end);
                privdata = data.substring(start, end);
            }
            if (data.contains("PUBLIC KEY")) {
                int pstart = data.indexOf("-----BEGIN PGP PUBLIC");
                int pend = data.indexOf("-----END PGP PUBLIC KEY BLOCK", pstart);
                pend = data.indexOf("\r", pend);
                pubdata = data.substring(pstart, pend);
            }
            out.close();
            input.close();
        }
        if (privdata.length() < 2 && pubdata.length() < 2) {
            LogAction.setFail();
            logger.info("Bad data");
            Core.sendFail(response, "Bad data");
            return;
        }
        Access acc = new Access();


        GuardKeys newkey = new GuardKeys();
        try { // If private key not empty, then import

            if (!privdata.equals("") && password != null && pgppassword != null) {
                try {
                    newkey = PgpKeys.ImportPGPPrivate(newkey, privdata, pgppassword, password);
                } catch (BadPasswordException ex) {
                    LogAction.setFail();
                    Core.sendFail(response, "Bad password");
                    return;
                }
            }
            if (newkey == null) {
                LogAction.setFail();
                Core.sendFail(response, "Failed to import private key");
                return;
            }
        } catch (Exception ex) {
            LogAction.setFail();
            logger.error("Unable to open pgp private file for inport " , ex);
            Core.sendFail(response, "Bad pgp");
            return;
        }
        if (pubdata.equals("")) {
            newkey = PgpKeys.createPubFromPrivate(newkey);

        } else {
            newkey.setPGPKeyRingFromAsc(pubdata);
            if (newkey.pubring == null) {
                LogAction.setFail();
                logger.info("Problem with pub keys");
                Core.sendFail(response, "Problem with pub keys");
                return;
            }
        }
        try {
            Api api = new Api(cookie,request);
            newkey.email = api.getPrimary();
            newkey.contextid = api.getContextId();
            newkey.userid = userid;
            //TODO:
            //language,version,question,answer,settings?


            String recovery = null;
            // Mkae sure the IDs of the new public key are same as the user
            if (verifyID (newkey)) {
                Long dupl;
                boolean duplicate = false;
                if ((dupl = isDuplicate(newkey)) != 0) {  // If public key already registered, possible duplicate
                    duplicate = true;
                }

                // Create recovery
                if (password != null) {
                    if (Keymanager.checkCreateRecovery(cookie, request)) {
                        newkey.recovery = Keymanager.createRecovery(newkey, password);
                    }
                }

                if (!duplicate) {  // Not duplicate, add
                    acc.addNewKey(newkey);
                } else {
                    GuardKeys existing = PGPPublicHandler.getKeyForPGP(userid, cid, dupl, true);

                    //////////// HANDLE MERGE/DUPLICATE public keys
                    if ((newkey.pubring = handleDuplicatePublic(newkey, existing, dupl)) == null) {
                        LogAction.setFail();
                        logger.error("Problem handling duplicate public key");
                        Core.sendFail(response, "duplicate error");
                        return;
                    }
                    if (newkey.getPGPSecretKeyRing() == null) {  // If overwriting public only, and we have a private for that key, keep it and only update PubRing
                        if (existing.getPGPSecretKeyRing() != null) {
                            OGKeyTableStorage ogKeyTableStorage = new RestDbOGKeyTableStorage();
                            ogKeyTableStorage.updatePublicKey(newkey);
                        }
                    } else {
                        // Update the key
                        newkey = updateOldGuardKey (newkey, existing, password);
                        acc.updateDuplPGPKey(userid, cid, newkey, recovery, dupl);
                    }
                }

            } else {
                LogAction.setFail();
                logger.info("Bad key id");
                Core.sendFail(response, "Bad ID");
                return;
            }

        } catch (Exception e) {
            LogAction.setFail();
            if (e.getMessage() != null) {
                if (e.getMessage().contains("Duplicate")) {
                    Core.sendFail(response, "duplicate");
                    logger.debug("Duplicate private key");
                    return;

                }
            }
            logger.error("Error adding pgp private key ", e);
            Core.sendFail(response, e.getMessage());
            return;
        }
        LogAction.setSuccess();
        Core.sendOK(response, "OK");
    }

    /**
     * Verify the public key has the same email as the user
     * @param gkey
     * @return
     */
    private boolean verifyID (GuardKeys gkey) {
        Iterator keys = gkey.pubring.getPublicKeys();
        PGPPublicKey key = null;
        while (keys.hasNext()) {
            key = (PGPPublicKey) keys.next();
            Iterator it = key.getUserIDs();
            int i = 0;
            JsonArray ids = new JsonArray();
            while (it.hasNext()) {
                String id = (String) it.next();
                if (id.toLowerCase().contains(gkey.email.toLowerCase())) {
                    return (true);
                }
            }
        }
        return (false);
    }

    /**
     * Handle an upload of a public key that already exists
     * @param gkey
     * @param dupl
     * @return

     */
    private PGPPublicKeyRing handleDuplicatePublic (GuardKeys gkey, GuardKeys existing, long dupl) {
        try {
        Iterator keys = gkey.pubring.getPublicKeys();
        // get the guardkeys that has the duplicated public key

        while (keys.hasNext()) {
            PGPPublicKey keyToImport = (PGPPublicKey) keys.next();
            long importID = keyToImport.getKeyID();
            PGPPublicKey toMerge = existing.getPGPPublicKey(importID);
            if (toMerge != null) {
                existing.pubring = PGPPublicKeyRing.removePublicKey(existing.pubring, toMerge);
                toMerge = mergeKeys (toMerge, keyToImport);
                if (toMerge == null) {
                    return (null);
                }
                existing.pubring = PGPPublicKeyRing.insertPublicKey(existing.pubring, toMerge);

            } else {
                existing.pubring = PGPPublicKeyRing.insertPublicKey(existing.pubring, keyToImport);
            }
        }
        PgpKeys.addPublicKeyIndex(existing.pubring, existing.email, existing.contextid, existing.keyid);
        return (existing.pubring);
        } catch (Exception e) {
            logger.error("Problem merging duplicate public keys, ", e);
            return (existing.pubring);
        }
    }

    /**
     * Since the salt changes, we need to re-encrypt the old Guard proprietary priv key.
     * @param newkey
     * @param existing
     * @param password
     * @return
     */
    private GuardKeys updateOldGuardKey (GuardKeys newkey, GuardKeys existing, String password) {
        if (existing.privKeyNull()) return (newkey);
        try {
            newkey.setPrivate(existing.getDecodedPrivate(password), password);
        } catch (Exception ex) {
            logger.error("Problem copying over old Guard Password ", ex);
            return (newkey);
        }
        return (newkey);
    }
    /**
     * Delete the private/public key pair for the user
     * @param request
     * @param response
     * @param ck
     * @throws Exception
     */
    public void DeletePrivateKey (HttpServletRequest request, HttpServletResponse response, OxCookie ck) throws Exception {
        JsonObject json = Core.getJSON(request);
        Api ap = new Api(ck, request);
        if (!ap.verifyLogin()) {
            Core.sendFail(response, "Must be logged in");
            LogAction.setFail();
            logger.info("Delete private key...not logged in");
            return;
        }
        Settings settings = Auth.getPermissions(ck, request);
        if (settings.noDeletePrivate) {
            Core.sendFail(response, "Function disabled by configuration");
            return;
        }
        int userid = json.get("userid").getAsInt();
        int cid = json.get("cid").getAsInt();
        String keystring = json.get("keyid").getAsString();
        String auth = json.get("auth").getAsString();
        long keyid = 0l;
        try {
            keyid = Long.parseLong(keystring);
        } catch (Exception e) {
            LogAction.setFail();
            logger.error("Bad keyid");
            Core.sendFail(response, "Bad keyid");
            return;
        }
        // Verify auth
        UserData data = new UserData(auth, ck);
        if (data.userid < 0) {
            Core.sendFail(response, "Bad password");
            logger.info("Bad password");
            return;
        }
        Access acc = new Access();
        GuardKeys key = acc.getKeys(userid, cid, keyid, false);
        // Make sure right password for this key
        if (!key.verifyPassword(data.encr_password)) {
            Core.sendFail(response, "Bad password");
            logger.info("Bad password");
            return;
        }
        boolean iscurrent = key.current;

        //TODO: use a service locator for resolvin this
        OGKeyTableStorage ogKeyTableStorage = new RestDbOGKeyTableStorage();
        //OK, password and login verified.  Delete the key
        ogKeyTableStorage.delete(key);

        //TODO: use a DI-Container to resolve this
        PGPKeysStorage pgpKeysStorage = new RestDbPGPKeysStorage();
        //Delete from
        pgpKeysStorage.deleteByKeyId(keyid);


        //if key was the current key, then upgrade highest version key to current
        if (iscurrent) {
        	resetCurrentKey(userid, cid);
        }
        Core.sendOK(response, "OK");
        logger.info("Success deleting private key");


    }

    /**
     * Resets the "current" key to the key which has the highest version
     * @param userid the user's id
     * @param cid the context id
     * @throws Exception
     */
    private void resetCurrentKey(int userid, int cid) throws Exception {
        //TODO: Using a Service Locator for resolving this
        OGKeyTableStorage ogKeyTableStorage = new RestDbOGKeyTableStorage();

        //Getting the keys with the highest version from the user, and mark it as the "current" key
        GuardKeys highestKey = ogKeyTableStorage.getHighestVersionKeyForUser(userid, cid);
        if(highestKey != null) {
            ogKeyTableStorage.setCurrentKey(highestKey);
        }
    }


    /**
     * Revoke a private key for the user
     * @param request
     * @param response
     * @param ck
     * @throws Exception
     */
    public void revokePrivateKey (HttpServletRequest request, HttpServletResponse response, OxCookie ck) throws Exception {
        JsonObject json = Core.getJSON(request);
        Api ap = new Api(ck, request);
        if (!ap.verifyLogin()) {
            Core.sendFail(response, "Must be logged in");
            LogAction.setFail();
            logger.info("Delete private key...not logged in");
            return;
        }
        int userid = json.get("userid").getAsInt();
        int cid = json.get("cid").getAsInt();
        String keystring = json.get("keyid").getAsString();
        String auth = json.get("auth").getAsString();
        String reason = json.get("reason").getAsString();
        long keyid = 0l;
        try {
            keyid = Long.parseLong(keystring);
        } catch (Exception e) {
            LogAction.setFail();
            logger.error("Bad keyid");
            Core.sendFail(response, "Bad keyid");
            return;
        }
        // Verify auth
        UserData data = new UserData(auth, ck);
        if (data.userid < 0) {
            Core.sendFail(response, "Bad password");
            logger.info("Bad password");
            return;
        }
        Access acc = new Access();
        GuardKeys key = acc.getKeys(userid, cid, keyid, false);
        // Make sure right password for this key
        if (!key.verifyPassword(data.encr_password)) {
            Core.sendFail(response, "Bad password");
            logger.info("Bad password");
            return;
        }
        key = PgpKeys.revoke(key, data.encr_password, reason);

        OGKeyTableStorage keyTableStorage = new RestDbOGKeyTableStorage();
        keyTableStorage.updatePublicKey(key);

        Core.sendOK(response, "OK");
        logger.info("Success deleting private key");

    }

    /**
     * Delete a local public key only that was uploaded by the user for his/her use
     * Private key must not exist
     * @param request
     * @param response
     * @param ck
     * @throws Exception
     */
    public void deleteLocalPublic (HttpServletRequest request, HttpServletResponse response, OxCookie ck) throws Exception {
        JsonObject json = Core.getJSON(request);
        Api ap = new Api(ck, request);
        if (!ap.verifyLogin()) {
            Core.sendFail(response, "Not logged in");
            logger.info("Fail, not logged in");
            return;
        }
        String keystring = Core.getStringFromJson(json, "keyid", true);
        long keyid = 0l;
        try {
            keyid = Long.parseLong(keystring);
        } catch (Exception ex) {
            Core.sendFail(response, "Bad keyid");
            logger.error("Bad keyid");
            return;
        }
        int userid = Core.getIntFromJson(json, "userid", true);
        int cid = Core.getIntFromJson(json, "cid", true);
        Access acc = new Access();
        GuardKeys key = acc.getKeys(userid, cid, keyid, false);
        if (key == null) {
            Core.sendFail(response, "Unable to find key");
            logger.info("Fail, unable to find key");
            return;
        }
        if (key.getPGPSecretKey() != null) {
            Core.sendFail(response, "Private key exists");
            logger.error("Attempt to delete public when private key exists");
            return;
        }

        //TODO: Use a service locator for resolving this
        OGKeyTableStorage ogKeyTableStorage = new RestDbOGKeyTableStorage();
        PGPKeysStorage pgpKeysStorage = new RestDbPGPKeysStorage();

        //Delete the key
        ogKeyTableStorage.delete(key);

        //delete the key mapping
        pgpKeysStorage.deleteByKeyIdInContext(keyid, cid);

        //if key was the current key, then upgrade highest version key to current:
        if (key.current) {
            resetCurrentKey(userid, cid);
        }
    }

    /**
     * Add/Merge certifications of two public keys
     * @param old
     * @param newKey
     * @return
     * @throws PGPException
     */
    private PGPPublicKey mergeKeys (PGPPublicKey old, PGPPublicKey newKey) throws PGPException {
        if (old.getKeyID() != newKey.getKeyID()) {
            logger.error("Unable to merge keys, different keyids ");
            return (null);
        }
        int sigcount = 0;
        ArrayList<PGPSignature> oldsigs = new ArrayList<PGPSignature> ();
        Iterator oldsigI = old.getSignatures();
        while (oldsigI.hasNext()) {
            oldsigs.add((PGPSignature) oldsigI.next());
        }

        Iterator<String> userids = old.getUserIDs();
        // Loop through userids of the old certificate and see if there are any new signatures
        while (userids.hasNext()) {
            String uid = userids.next();
            Iterator<PGPSignature> newsigs = newKey.getSignaturesForID(uid);
            if (newsigs != null) {
                while (newsigs.hasNext()) {
                    PGPSignature newsig = newsigs.next();
                    if (!hasSig (newsig, oldsigs)) {
                        old = PGPPublicKey.addCertification(old, uid, newsig);
                        sigcount++;
                    }
                }
            }

        }
        userids = newKey.getUserIDs();
        // Loop through userids of the NEW certificate and see if there are any new signatures
        while (userids.hasNext()) {
            String uid = userids.next();
            Iterator<PGPSignature> newsigs = newKey.getSignaturesForID(uid);
            while (newsigs.hasNext()) {
                PGPSignature newsig = newsigs.next();
                if (!hasSig (newsig, oldsigs)) {
                    old = PGPPublicKey.addCertification(old, uid, newsig);
                    sigcount++;
                }
            }
        }
        // Loop through user attributes and see if any new signatures
        Iterator<PGPUserAttributeSubpacketVector> vectors = old.getUserAttributes();
        while (vectors.hasNext()) {
            PGPUserAttributeSubpacketVector vector = vectors.next();
            Iterator<PGPSignature> newsigs = newKey.getSignaturesForUserAttribute(vector);
            while (newsigs.hasNext()) {
                PGPSignature newsig = newsigs.next();
                if (!hasSig (newsig, oldsigs)) {
                    old = PGPPublicKey.addCertification(old, vector, newsig);
                    sigcount++;
                }
            }
        }
        // Loop through user attributes and see if any new signatures
         vectors = newKey.getUserAttributes();
        while (vectors.hasNext()) {
            PGPUserAttributeSubpacketVector vector = vectors.next();
            Iterator<PGPSignature> newsigs = newKey.getSignaturesForUserAttribute(vector);
            if (newsigs != null) {
                while (newsigs.hasNext()) {
                    PGPSignature newsig = newsigs.next();
                    if (!hasSig (newsig, oldsigs)) {
                        old = PGPPublicKey.addCertification(old, vector, newsig);
                        sigcount++;
                    }
                }
            }

        }

        // Check for revoke signatures
        Iterator<PGPSignature> newsigs = newKey.getSignaturesOfType(PGPSignature.KEY_REVOCATION);
        while (newsigs.hasNext()) {
            PGPSignature newsig = newsigs.next();
            if (!hasSig (newsig, oldsigs)) {
                old = PGPPublicKey.addCertification(old, newsig);
                sigcount++;
            }
        }
        return (old);
    }

    private boolean hasSig (PGPSignature sig, ArrayList<PGPSignature> sigs) throws PGPException {
        for (int i = 0; i < sigs.size(); i++) {
            if (Arrays.equals(sig.getSignature(), sigs.get(i).getSignature())) {
                return (true);
            }
        }
        return (false);
    }

    /**
     * Check if public key already exists in our index
     * @param gkey
     * @return 0 if not duplicate, otherwise keyid of the duplicate key
     * @throws Exception
     */
    private long isDuplicate (GuardKeys gkey) throws Exception {
       List<Long> ids = new ArrayList<Long>();
       Iterator<PGPPublicKey> keys = gkey.pubring.getPublicKeys();
       while (keys.hasNext()) {
           ids.add(keys.next().getKeyID());
       }

       //TODO: Use a DI Container for resolving this
       PGPKeysStorage pgpKeysStorage = new RestDbPGPKeysStorage();
       List<PGPKeys> keyMappings = pgpKeysStorage.getByIds(ids);
       if(keyMappings.size() > 0) {
           return keyMappings.get(0).getKeyId();
       }
       return 0;
    }

    /**
     * Make the keyid the current key
     * @param id
     * @param cid
     * @param keyid
     * @return
     * @throws Exception
     */
    public static void makeCurrent (int id, int cid, String keyid) throws Exception {
        //TODO: use a Service Locator for resolving this
        OGKeyTableStorage ogKeyTableStorage = new RestDbOGKeyTableStorage();

        //Getting the key which should be marked as "current"
        GuardKeys key = ogKeyTableStorage.getKeyForUserById(id, cid, keyid);
        if(key != null) {
            //Removing the current flag from all keys
            ogKeyTableStorage.unsetCurrentFlag(id, cid);

            //Setting the current flag to the key
            ogKeyTableStorage.setCurrentKey(key);
        }
        else {
            throw new InvalidArgumentException("Key Not Found");
        }
    }

    /**
     * Make a selected Private/Public key pair the current one to be used.
     * @param request
     * @param response
     * @param ck
     * @throws Exception
     */
    public void makeCurrent (HttpServletRequest request, HttpServletResponse response, OxCookie ck) throws Exception {
        Api ap = new Api(ck, request);
        if (!ap.verifyLogin()) {
            Core.sendFail(response, "Must be logged in");
            return;
        }
        int userid = Core.getIntParameter(request, "userid", true);
        int cid = Core.getIntParameter(request, "cid", true);
        LogAction.setUser(userid, cid);
        String keyid = Core.getStringParameter(request, "keyid", true);
        makeCurrent (userid, cid, keyid);
    }

    public void addUID (HttpServletRequest request, HttpServletResponse response, OxCookie ck) throws Exception {
        JsonObject json = Core.getJSON(request);
        Api ap = new Api(ck, request);
        if (!ap.verifyLogin()) {
            Core.sendFail(response, "Must be logged in");
            return;
        }
        int userid = Core.getIntParameter(request, "userid", true);
        int cid = Core.getIntParameter(request, "cid", true);
        LogAction.setUser(userid, cid);
        String keyid = Core.getStringParameter(request, "keyid", true);
        String name = Core.getStringFromJson(json, "name", true);
        String email = Core.getStringFromJson(json, "email", true);
        String password = Core.getStringFromJson(json, "password", true);
        Access acc = new Access();
        GuardKeys keys = acc.getKeys(userid, cid, Long.parseLong(keyid), false);
        PGPPrivateKey pkey;
        try {
            pkey = keys.getPGPSignKey(password);
        } catch (PGPException ex) {
            Core.sendFail(response, "Bad password");
            return;
        }
        String uid = name + " <" + email + ">";
        PGPPublicKeyRing newring = PgpKeys.addUID(pkey, keys.pubring, uid);
        if (newring == null) {
            Core.sendFail(response, "error adding");
            return;
        }
        keys.pubring = newring;
        acc.updatePGPKeys(userid, cid, keys);
        Core.sendOK(response, "OK");
    }


}
