package com.openexchange.guard.database.ogKeyTable;

import java.util.ArrayList;
import java.util.List;
import com.amazonaws.services.cloudfront_2012_03_15.model.InvalidArgumentException;
import com.openexchange.guard.database.DbCommand;
import com.openexchange.guard.database.DbQuery;
import com.openexchange.guard.database.OxResultSet;
import com.openexchange.guard.database.Sharding;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.pgp.PGPPublicKeyRingExporter;

public class RestDbOGKeyTableStorage implements OGKeyTableStorage {

    /**
     * Factory method to create a GuardKey from a database result set
     * @param rs the result set to create the key from
     * @return A GuardKeys object created from the given result set
     * @throws Exception due an error
     */
    private GuardKeys createKey (OxResultSet rs) throws Exception {
        GuardKeys key = new GuardKeys();
        key.setPGPSecretFromString(rs.getString("PGPSecret"));
        key.setPGPKeyRingFromAsc(rs.getString("PGPPublic"));
        key.setEncodedPrivate(rs.getString("RSAPrivate"));
        key.setPublicKeyFrom64String(rs.getString("RSAPublic"));
        key.setSalt(rs.getString("Salt"));
        key.misc = rs.getString("misc");
        key.settings = rs.getJson("settings");
        key.email = rs.getString("email");
        key.lastup = rs.getDate("lastMod");
        key.question = rs.getString("question");
        key.answer = rs.getString("answer");
        key.contextid = rs.getInt("cid");
        key.userid = rs.getInt("id");
        key.keyid = rs.getLong("keyid");
        key.version = rs.getInt("version");
        key.current = rs.getBit("current");
        key.language = rs.getString("lang");
        key.recovery = rs.getString("Recovery");
        key.recoveryAvail = key.recovery == null ? false : !key.recovery.equals("");
        if (key.recovery != null) {
            if (key.recovery.equals("-1") || (key.lastup == null)) {
                key.passwordNeeded = true;
            }
        }
        return (key);
    }

    @Override
    public List<GuardKeys> getKeysForUser(int userId, int contextId) throws Exception {
        DbQuery db = new DbQuery() ;
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.SELECT_FOR_USER_STMT);
            command.addVariables(userId);
            command.addVariables(contextId);
            db.read(command, userId, contextId);
            ArrayList<GuardKeys> ret = new ArrayList<GuardKeys>();
            while(db.next()) {
                ret.add(createKey(db.rs));
            }
            return ret;
        }
        finally {
            db.close();
        }
    }

    @Override
    public GuardKeys getKeyForUserById(int userId, int contextId, String keyid) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.SELECT_FOR_USER_BY_KEYID_STMT);
            command.addVariables(userId);
            command.addVariables(contextId);
            command.addVariables(keyid);
            db.read(command, userId, contextId);
            if(db.next()) {
                return createKey(db.rs);
            }
            return null;
        }
        finally {
            db.close();
        }
    }


    @Override
    public GuardKeys getKeyForUserById(int userId, int contextId, int shardId, String keyId) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.SELECT_FOR_USER_BY_KEYID_STMT);
            command.addVariables(userId);
            command.addVariables(contextId);
            command.addVariables(keyId);
            if (contextId >= 0) {
                if (userId < 0) {// minus 1 is the recovery password, which is in OG
                    db.readOG(command);
                } else {
                    db.readFromDB(command, contextId);
                }
            } else {
                db.readFromShard(command, shardId);
            }

            if(db.next()) {
                return createKey(db.rs);
            }
            return null;
        }finally {
            db.close();
        }
    }

    @Override
    public GuardKeys getHighestVersionKeyForUser(int userId, int contextId) throws Exception{
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.SELECT_HIGHEST_VERSION_KEY_STMT);
            command.addVariables(contextId);
            command.addVariables(userId);
            command.addVariables(contextId);
            command.addVariables(userId);
            db.read(command, userId, contextId);
            if(db.next()) {
                return createKey(db.rs);
            }
            return null;
        }
        finally {
            db.close();
        }
    }

    @Override
    public List<GuardKeys> getKeysForContext(int contextId) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.SELECT_FOR_CONTEXT_STMT);
            command.addVariables(contextId);
            db.readFromDB(command, contextId);
            ArrayList<GuardKeys> ret = new ArrayList<GuardKeys>();
            while(db.next()) {
                ret.add(createKey(db.rs));
            }
            return ret;
        }
        finally {
            db.close();
        }
    }

    @Override
    public GuardKeys getKeyForEmailAndContext(String email, int contextid) throws Exception {
        int shard = 0;
        if (contextid < 0) { // If guest, get shard number, then check tables exist
            shard = Sharding.getShard(email);
            if (shard == 0) {
                return null;
            }
            Sharding.checkTables(shard, false);
        } else {
            if (contextid > 0)
             {
                Sharding.checkTables(contextid, true); // Check these OG tables exist
            }
        }
        return (getKeyForEmailAndContext(email, contextid, shard));
    }
    @Override
    public GuardKeys getKeyForEmailAndContext(String email, int contextid, int shard) throws Exception {
        DbQuery db = new DbQuery();
        try{
            DbCommand command = new DbCommand(OGKeyTableSql.SELECT_FOR_EMAIL_IN_CONTEXT_STMT);
            command.addVariables(email);

            if (contextid >= 0) {
                if (contextid > 0) {
                    db.readFromDB(command, contextid);
                }
                else {
                    db.readOG(command);
                }
            }
            else {
                db.readFromDB(command, Sharding.database_name(shard));
            }
            if (db.next()) {
                return createKey(db.rs);
            }
            return null;
        }
        finally {
            db.close();
        }
    }

    @Override
    public GuardKeys getCurrentKeyForUser(int userId, int contextId, String email) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.SELECT_CURRENT_FOR_USER_EMAIL_STMT);
            command.addVariables(email);
            db.read(command, userId, contextId);
            if(db.next()) {
                return createKey(db.rs);
            }
            return null;
        }
        finally {
            db.close();
        }
    }

    @Override
    public GuardKeys getCurrentKeyForUser(int userId, int contextId) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.SELECT_CURRENT_FOR_USER_STMT);
            command.addVariables(userId);
            command.addVariables(contextId);
            db.read(command, userId, contextId);
            if(db.next()) {
                return createKey(db.rs);
            }
            return null;
        }
        finally {
            db.close();
        }
    }

    @Override
    public GuardKeys getCurrentKeyForUser(int userId, int contextId, int shardId) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.SELECT_CURRENT_FOR_USER_STMT);
            command.addVariables(userId);
            command.addVariables(contextId);
            if (contextId >= 0) {
                if (userId < 0) {// minus 1 is the recovery password, which is in OG
                    db.readOG(command);
                } else {
                    db.readFromDB(command, contextId);
                }
            } else {
                db.readFromShard(command, shardId);
            }

            if(db.next()) {
                return createKey(db.rs);
            }
            return null;
        }
        finally {
            db.close();
        }
    }


    @Override
    public GuardKeys getGuardDaemonKey() throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.SELECT_GUARD_DAEMON_KEY_STMT);
            db.readOG(command);
            if(db.next()) {
                return createKey(db.rs);
            }
            return null;
        }
        finally {
            db.close();
        }
    }
    
    @Override
    public void insert (GuardKeys key,boolean setLastMod) throws Exception {
        insert (key, setLastMod, 0);
    }

    @Override
    public void insert(GuardKeys key,boolean setLastMod, int shard) throws Exception {

        DbQuery db = new DbQuery();
        try {
            String sql = setLastMod ?
                         OGKeyTableSql.INSERT_AND_SET_LASTMOD_STMT :
                         OGKeyTableSql.INSERT_STMT;
            DbCommand command = new DbCommand(sql);
            command.addVariables((key.userid == 0 ? null : key.userid));
            command.addVariables(key.contextid);
            command.addVariables(key.getEncodedPGPSecret());
            command.addVariables(PGPPublicKeyRingExporter.export(key.pubring));
            command.addVariables(key.getEncodedPrivate());
            command.addVariables(key.getEncodedPublic());
            command.addVariables(key.recovery);
            command.addVariables(key.getSalt());
            command.addVariables(key.email);
            command.addVariables(key.language);
            command.addVariables(key.pubring.getPublicKey().getKeyID());
            command.addVariables(key.version);
            command.addVariables(key.question);
            command.addVariables(key.answer);
            command.addVariables(key.settings == null ? null : key.settings.toString());
            command.addVariables(key.current);

            //OX users go in OX database, otherwise guest shard
            if (key.contextid >= 0) {
                if(key.userid < 0) {
                    db.writeOxGuard(command);
                }
                else {
                    db.writeToDB(command, key.contextid);
                }
            } else {
                if (shard == 0) shard = Sharding.getNextShard();
                if(shard > 0) {
                    Sharding.checkTables(shard, false);
                    db.writeToDb(command, Sharding.database_name(shard));
                } else {
                    throw new Exception ("Unable to get new Guest shard");
                }
            }
        }
        finally {
            db.close();
        }
    }

    @Override
    public void updatePublicKey(GuardKeys key) throws Exception {
        if(key.pubring == null) {
            throw new InvalidArgumentException("missing pubring in parameter key");
        }

        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.UPDATE_PGP_PUBLIC_STMT);
            command.addVariables(PGPPublicKeyRingExporter.export(key.pubring));
            command.addVariables(key.keyid);
            db.write(command, key.userid, key.contextid);
        }
        finally {
            db.close();
        }
    }

    @Override
    public void updatePublicAndPrivateKey(GuardKeys key) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.UPDATE_PGP_PUBLIC_AND_PRIVATE_STMT);
            command.addVariables(key.getEncodedPGPSecret());
            command.addVariables(PGPPublicKeyRingExporter.export(key.pubring));
            command.addVariables(key.userid);
            command.addVariables(key.contextid);
            command.addVariables(key.keyid);
            db.write(command, key.userid, key.contextid);
        }
        finally {
            db.close();
        }
    }

    @Override
    public void updatePassword(GuardKeys key, String recovery, boolean setLastMod) throws Exception {

        if(key.keyid == 0) {
            throw new IllegalArgumentException("keyid is missing");
        }

        DbQuery db = new DbQuery();
        try {
            String sql = setLastMod ? OGKeyTableSql.UPDATE_PASSWORDS_AND_LASTMOD_STMT :
                                      OGKeyTableSql.UPDATE_PASSWORDS_STMT;
            DbCommand command = new DbCommand(sql);
            command.addVariables(key.getEncodedPGPSecret());
            command.addVariables(key.getEncodedPrivate());
            command.addVariables(recovery);
            command.addVariables(key.userid);
            command.addVariables(key.contextid);
            command.addVariables(key.keyid);
            db.write(command, key.userid, key.contextid);
        }
        finally {
            db.close();
        }
    }

    @Override
    public void updatePin(int userId, int contextId, String newPin) throws Exception{
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.UPDATE_PIN_STMT);
            command.addVariables(newPin);
            command.addVariables(userId);
            command.addVariables(contextId);
            db.write(command, userId, contextId);
        }
        finally {
            db.close();
        }
    }

    @Override
    public void updateDuplicate(GuardKeys key, String recovery, Long updateDuplicatedKeyId) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.UPDATE_DUPLICATED_STMT);
            command.addVariables(key.getEncodedPrivate());
            command.addVariables(key.getEncodedPGPSecret());
            command.addVariables(PGPPublicKeyRingExporter.export(key.pubring));
            command.addVariables(recovery);
            command.addVariables(key.getSalt());
            command.addVariables(key.userid);
            command.addVariables(key.contextid);
            command.addVariables(updateDuplicatedKeyId);
        }
        finally {
            db.close();
        }
    }

    @Override
    public void updateQuestionForUser(int userId, int contextId, String question, String answer) throws Exception{
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.UPDATE_QUESTION_STMT);
            command.addVariables(question);
            command.addVariables(answer);
            command.addVariables(userId);
            command.addVariables(contextId);
            db.write(command, userId, contextId);
        }
        finally {
            db.close();
        }
    }

    @Override
    public void updateKeyVersion(GuardKeys key, int newVersion) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.UPDATE_VERSION_STMT);
            command.addVariables(newVersion);
            command.addVariables(key.contextid);
            command.addVariables(key.userid);
            command.addVariables(key.version);
            db.write(command, key.userid, key.contextid);
        }
        finally {
            db.close();
        }

    }

    @Override
    public void setCurrentKey(GuardKeys key) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.SET_CURRENT_STMT);
            command.addVariables(key.userid);
            command.addVariables(key.contextid);
            command.addVariables(key.keyid);
            db.write(command, key.userid, key.contextid);
        }
        finally {
            db.close();
        }
    }

    @Override
    public void unsetCurrentFlag(int userId, int contextId) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.UNSET_CURRENT_FOR_USER_STMT);
            command.addVariables(userId);
            command.addVariables(contextId);
            // OX users go in OX database, otherwise guest shard
            if(contextId >= 0) {
                db.writeToDB(command, contextId);
            }
            else {
                db.write(command, userId, contextId);
            }
        }
        finally {
            db.close();
        }
    }

    @Override
    public void delete(GuardKeys key) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.DELETE_KEY_STMT);
            command.addVariables(key.contextid);
            command.addVariables(key.userid);
            command.addVariables(key.keyid);
            db.write(command, key.userid, key.contextid);
        }
        finally {
            db.close();
        }
    }

    @Override
    public void deleteAllForUser(int userId, int contextId) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.DELETE_FOR_USER_STMT);
            command.addVariables(userId);
            command.addVariables(contextId);
            db.write(command, userId, contextId);
        }
        finally {
            db.close();
        }
    }

    @Override
    public void deleteAllForContext(int contextId) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.DELETE_FOR_CONTEXT_STMT);
            command.addVariables(contextId);
            db.writeToDB(command, contextId);
        }
        finally {
            db.close();
        }
    }

    @Override
    public void deleteRecovery(GuardKeys key) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.DELETE_RECOVERY_FROM_KEY_STMT);
            command.addVariables(key.userid);
            command.addVariables(key.contextid);
            db.write(command, key.userid, key.contextid);
        }
        finally {
            db.close();
        }
    }

    @Override
    public boolean exists(int contextId) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.EXISTS_STMT);
            db.readFromDB(command, contextId);
            return true;
        }
        catch(Exception e) {
            if (e.getMessage().contains("doesn't exist")) {
                return false;
            }
            throw e;
        }
        finally {
            db.close();
        }
    }

    @Override
    public boolean existsInShard(String shardDatabaseName) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.EXISTS_STMT);
            db.readFromDB(command, shardDatabaseName);
            return true;
        }
        catch(Exception e) {
            if (e.getMessage().contains("doesn't exist")) {
                return false;
            }
            throw e;
        }
        finally {
            db.close();
        }
    }

    @Override
    public void updateAutoIncrementInShard(String shardDatabaseName, int value) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGKeyTableSql.ALTER_AUTOINCREMENT_STMT);
            command.addVariables(value);
            db.writeToDb(command, shardDatabaseName);
        }
        finally {
            db.close();
        }
    }
}
