package com.openexchange.guard.database.keyCache;

import com.openexchange.guard.database.DbCommand;
import com.openexchange.guard.database.DbQuery;

/**
 * {@link RestDbKeyCacheStorage} provides a REST DB implementation of the KeyCacheStorage
 */
public class RestDbKeyCacheStorage implements KeyCacheStorage {

    /**
     * {@link PreGeneratedKey} represents a pre generated key stored in the Database
     */
    private class PreGeneratedKey {

        private final int id;
        private final String data;

        public PreGeneratedKey(int id, String data) {
            super();
            this.id = id;
            this.data = data;
        }

        public int getId() {
            return id;
        }

        public String getData() {
            return data;
        }
    }

    /**
     * Internal method to get the next key from the DB
     * @return the next key or null if no more keys are available
     */
    private PreGeneratedKey getNextKey() throws Exception{
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(KeyCacheSql.NEXT_STMT);
            db.readOG(command);
            if(db.next()) {
                return new PreGeneratedKey(db.rs.getInt("id"),
                               db.rs.getString("privkey"));
            }
            return null;
        }
        finally {
            db.close();
        }
    }

    /**
     * Deletes a pre generated key from the database
     * @param key the key to delete
     * @throws Exception due an error
     */
    private void deleteKey(PreGeneratedKey key) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(KeyCacheSql.DELETE_STMT);
            command.addVariables(key.getId());
            db.writeOxGuard(command);
        }
        finally {
            db.close();
        }
    }

    @Override
    public void insert(String keyData) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(KeyCacheSql.INSERT_STMT);
            command.addVariables(keyData);
            db.writeOxGuard(command);
        }
        finally {
            db.close();
        }
    }

    @Override
    public String popNext() throws Exception {
        PreGeneratedKey key = getNextKey();
        if(key != null) {
            deleteKey(key);
            return key.getData();
        }
        return null;
    }

    @Override
    public int getCount() throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(KeyCacheSql.COUNT_STMT);
            db.readOG(command);
            if(db.next()) {
                return db.rs.getInt("count");
            }
            return 0;
        }
        finally {
            db.close();
        }
    }
}
