package com.openexchange.guard.database.deletedKey;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.openexchange.guard.database.DbCommand;
import com.openexchange.guard.database.DbQuery;

/**
 * {@link RestDbDeletedKeysStorage} provides a REST DB implementation of the DeletedKeysStorage
 */
public class RestDbDeletedKeysStorage implements DeletedKeysStorage {

    private static Logger logger = LoggerFactory.getLogger(RestDbDeletedKeysStorage.class);

    @Override
    public void insert(DeletedKey... deletedKeys) throws Exception {
        DbQuery dbQuery = null;
        try {

            for (int i = 0; i < deletedKeys.length; i++) {
                dbQuery = new DbQuery();
                DeletedKey deletedKey = deletedKeys[i];
                DbCommand command = new DbCommand(DeletedKeysSql.INSERT_KEY_STMT);
                command.addVariables(deletedKey.getUserId());
                command.addVariables(deletedKey.getCid());
                command.addVariables(deletedKey.getEmail());
                command.addVariables(deletedKey.getPGPSecret());
                command.addVariables(deletedKey.getSalt());
                command.addVariables(deletedKey.getRecovery());
                command.addVariables(deletedKey.getVersion());
                dbQuery.writeOxGuard(command);
            }
        } catch (Exception e) {
            logger.error(
                String.format("Error while inserting into table deletedKeys: %s",
                              e.getMessage()));
            e.printStackTrace();
            throw e;
        }
        finally {
            if(dbQuery != null) {
                dbQuery.close();
            }
        }

    }

    @Override
    public void setExposedForEmail(String email, int cid) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(DeletedKeysSql.UPDATE_KEY_SET_EXPOSED_STMT);
            SimpleDateFormat sdf = new java.text.SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            String exposedDateString = sdf.format(new Date());
            command.addVariables(exposedDateString);
            command.addVariables(email);
            command.addVariables(cid);
            db.writeOxGuard(command);
        }
        catch (Exception e) {
            logger.error(
                String.format("Error while exposing keys for email %s: %s",
                              email,
                              e.getMessage()));
            e.printStackTrace();
            throw e;
        }
        finally {
            db.close();
        }
    }

    @Override
    public DeletedKey getFirstForEmail(String email, int cid) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(DeletedKeysSql.SELECT_KEY_BY_EMAIL_CID_STMT);
            command.addVariables(email);
            command.addVariables(cid);

            db.readOG(command);
            if(db.next()) {
                return new DeletedKey(db.rs.getInt("id"),
                                      db.rs.getInt("cid"),
                                      db.rs.getString("PGPSecret"),
                                      db.rs.getString("recovery"),
                                      db.rs.getString("salt"),
                                      db.rs.getInt("version"),
                                      db.rs.getString("email"),
                                      db.rs.getBit("exposed"));

            }
            else {
                return null;
            }
        } catch (Exception e) {
            logger.error(String.format("Could not get deleted key for email %s in context %d",
                                       email,
                                       cid),
                         e);
            throw e;
        }
        finally{
            db.close();
        }
    }

    @Override
    public int setAllUnexposed(Date before) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(DeletedKeysSql.UPDATE_KEY_SET_UNEXPOSED_BY_DATE_STMT);
            SimpleDateFormat sdf = new java.text.SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            String beforeDate = sdf.format(before);
            command.addVariables(beforeDate);
            JsonObject response = db.writeOxGuard(command);
            JsonObject results = response.get("results").getAsJsonObject();

            //Getting the amount of keys which were updated by the DB-command
            int updatedCount = -1;
            for (Map.Entry<String,JsonElement> entry : results.entrySet()) {
                JsonElement elem = entry.getValue();
                JsonObject result = elem.getAsJsonObject();
                if(result.has("updated")){
                    updatedCount = result.get("updated").getAsInt();
                    break;
                }
            }

            if(updatedCount > -1) {
                return updatedCount;
            }
            else {
                throw new Exception("Invalid updated response from SQL update statement.");
            }
        }
        finally {
            db.close();
        }
    }

    @Override
    public ArrayList<DeletedKey> getAllExposedForEmail(String email, int id, int cid) throws Exception {
        DbQuery db = new DbQuery();
        try {
            ArrayList<DeletedKey> keys = new ArrayList<DeletedKey> ();
            DbCommand com = new DbCommand(DeletedKeysSql.SELECT_KEY_EXPOSED_BY_EMAIL_CID_STMT);
            com.addVariables(cid);
            com.addVariables(email);
            com.addVariables(id);

            db.readOG(com);
            while (db.next()) {
                DeletedKey key = new DeletedKey(db.rs.getInt("id"),
                                                db.rs.getInt("cid"),
                                                db.rs.getString("PGPSecret"),
                                                db.rs.getString("recovery"),
                                                db.rs.getString("salt"),
                                                db.rs.getInt("version"),
                                                db.rs.getString("email"),
                                                db.rs.getBit("exposed"));
                keys.add(key);
            }
            return keys;
        } catch (Exception e) {
            logger.error(
                String.format("Error while reading exposed keys for email %s: %s",
                              email,
                              e.getMessage()));
            e.printStackTrace();
            throw e;
        }
        finally {
            db.close();
        }
    }
}
