package com.openexchange.guard.database.deletedKey;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import org.apache.commons.codec.binary.Base64;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.operator.bc.BcKeyFingerprintCalculator;
import org.bouncycastle.openpgp.operator.jcajce.JcePBESecretKeyDecryptorBuilder;
import com.openexchange.guard.encr.EncrLib;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.exceptions.BadPasswordException;
import com.openexchange.guard.pgp.PgpHandler;
import com.openexchange.guard.pgp.PgpKeys;

/**
 * Class for a key that was deleted and saved in the deletedKeyTable
 * @author greg
 *
 */
public class DeletedKey {

    private final int userid;
    private final int cid;
    private PGPSecretKeyRing secretKeyRing;
    private String pgpSecret;
    private String recovery;
    private String salt;
    private int version;
    private String email;
    private boolean exposed;

    /**
     * Creating a deleted key object based on a existing key
     * @param key The key to create the deleted key from
     */
    public DeletedKey(GuardKeys key) throws Exception {
        this.userid = key.userid;
        this.cid = key.contextid;
        this.secretKeyRing = new PGPSecretKeyRing(Base64.decodeBase64(key.getEncodedPGPSecret()), new BcKeyFingerprintCalculator());
        this.recovery = key.recovery;
        this.salt = key.getSalt();
        this.version = key.version;
        this.email = key.email;
        this.pgpSecret = key.getEncodedPGPSecret();
    }

    public DeletedKey(int userId,
                      int cid,
                      String pgpSecret,
                      String recovery,
                      String salt,
                      int version,
                      String email,
                      boolean exposed ) throws Exception, PGPException {
        this.userid = userId;
        this.cid = cid;
        this.secretKeyRing = new PGPSecretKeyRing(Base64.decodeBase64(pgpSecret), new BcKeyFingerprintCalculator());
        this.pgpSecret = pgpSecret;
        this.recovery = recovery;
        this.salt = salt;
        this.version = version;
        this.email = email;
        this.exposed = exposed;
    }

    public PGPSecretKeyRing getSecretKeyRing() {
        return secretKeyRing;
    }

    public void setSecretKeyRing(PGPSecretKeyRing secretKeyRing) {
        this.secretKeyRing = secretKeyRing;
    }

    public String getRecovery() {
        return recovery;
    }

    public void setRecovery(String recovery) {
        this.recovery = recovery;
    }

    public String getSalt() {
        return salt;
    }

    public void setSalt(String salt) {
        this.salt = salt;
    }

    public int getVersion() {
        return version;
    }

    public void setVersion(int version) {
        this.version = version;
    }

    public String getEmail() {
        return email;
    }

    public void setEmail(String email) {
        this.email = email;
    }

    public String getPGPSecret() {
        return pgpSecret;
    }

    public void setPgpSecret(String pgpSecret) {
        this.pgpSecret = pgpSecret;
    }

    public boolean isExposed() {
        return exposed;
    }

    public void setExposed(boolean exposed) {
        this.exposed = exposed;
    }

    public int getUserId() {
        return userid;
    }

    public int getCid() {
        return cid;
    }

    /**
     * Verify the password is correct for this key
     * @param password
     * @return
     */
    public boolean verifyPassword (String password) {
        char [] pass = EncrLib.getSHA(password, salt).toCharArray();
        PGPSecretKey sec_key = secretKeyRing.getSecretKey();
        try {
            PGPPrivateKey decryptor = sec_key.extractPrivateKey(new JcePBESecretKeyDecryptorBuilder().setProvider("BC").build(pass));
            if (decryptor != null) {
                return(true);
            }
        } catch (PGPException ex) {
            return (false);
        }
        return (false);
    }

    /**
     * Export the private key using the supplied password
     * @param password
     * @return
     * @throws PGPException
     * @throws BadPasswordException
     * @throws IOException
     */
    public String export (String password) throws PGPException, BadPasswordException, IOException {
        GuardKeys key = new GuardKeys(this);
        String export = PgpKeys.ExportPGPPrivate(key, password);
        return (export);
    }

    /**
     * Get arraylist of userids for this keyring
     * @return
     */
    public ArrayList<String> getUserIds() {
        ArrayList<String> ids = new ArrayList<String> ();
        Iterator<String> it = secretKeyRing.getSecretKey().getUserIDs();
        while (it.hasNext()) {
            ids.add(it.next());
        }
        return (ids);
    }

    /**
     * Get arraylist of fingerprints for this keyring
     * @return
     */
    public ArrayList<String> getKeyFingerprints () {
        ArrayList<String> fps = new ArrayList<String> ();
        Iterator<PGPSecretKey> keys = secretKeyRing.getSecretKeys();
        while (keys.hasNext()) {
            PGPSecretKey skey = keys.next();
            fps.add(PgpHandler.getFingerPrintString(skey.getPublicKey().getFingerprint()));
        }
        return(fps);
    }
}
