package com.openexchange.guard.translate;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * {@link ConditionalVissibleTemplateTransformation} allows to define conditional elements in templates.
 *
 * <p>Put the following pattern in your template in order define conditional content: #if(name)..#endif
 *
 * <p>If the condition passed to the constructor is true, {@link ConditionalVissibleTemplateTransformation}
 * will remove the outer parts of the pattern, which are #if(name) and #endif, while the inner text will be remaining.
 * If the condition is false, the whole pattern including the inner text will be removed.
 */
public class ConditionalVissibleTemplateTransformation implements TemplateTransformation {

    private final String name;
    boolean transform;

    /**
     *
     * Initializes a new {@link ConditionalVissibleTemplateTransformation}.
     * @param name The name of the condition
     * @param condition true, if the inner text of the condition pattern should remain, false, if the whole pattern should be removed
     */
    public ConditionalVissibleTemplateTransformation(String name, boolean condition){
        this.name = name;
        this.transform = condition;
    }

    @Override
    public String transformTemplate(String template) {
        String regex = "#if\\((" + name + ")\\)(.*?)(#endif)";
        Pattern pattern = Pattern.compile(regex,Pattern.DOTALL);
        Matcher matcher = pattern.matcher(template);
        while(matcher.find()) {
            if(matcher.group(1).equals(name)) {
                if(transform) {
                    //removing everything
                    template = matcher.replaceAll("");
                }
                else {
                    //Only keeping the middle group (without the #if and #endif groups)
                    template = matcher.replaceAll("$2");
                }
            }
        }
        return template;
    }
}
