package com.openexchange.guard.storage.s3;

import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.amazonaws.AmazonClientException;
import com.amazonaws.services.s3.model.AbortMultipartUploadRequest;
import com.amazonaws.services.s3.model.CompleteMultipartUploadRequest;
import com.amazonaws.services.s3.model.InitiateMultipartUploadRequest;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PartETag;
import com.amazonaws.services.s3.model.UploadPartRequest;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.encr.EncryptedObj;
import com.openexchange.guard.storage.Storage;

/**
 * Class for saving encrypted objects to and from an S3 object store.
 * 
 * @author benjamin.otterbach@open-xchange.com
 */
public class S3Storage {

	private static Logger logger = LoggerFactory.getLogger(S3Storage.class);
    /**
     * The size of the in-memory buffer for uploads to use.
     */
    private static final int UPLOAD_BUFFER_SIZE = 1024 * 1024 * 2;
    
    /**
     * Write byte array to filename
     * 
     * @param filename Full filename of object
     * @param data byte array of data
     * @return
     * @throws IOException
     */
    public static Boolean writeObj(String directoryPrefix, String ObjId, byte[] data) {
    	boolean uploadSucess = false;
    	
        /*
         * prepare upload
         */
        try {
//            if (data.length < UPLOAD_BUFFER_SIZE) {
                /*
                 * whole file fits into buffer (this includes a zero byte file), upload directly
                 */
                ByteArrayInputStream dataStream = null;
                try {
                	dataStream = new ByteArrayInputStream(data);
                    ObjectMetadata metadata = new ObjectMetadata();
                    metadata.setContentLength(data.length);
                    S3ClientFactory.s3Client.putObject(Config.getS3BucketName(), directoryPrefix + Storage.DELIMITER + ObjId, dataStream, metadata);
                    uploadSucess = true;
                } finally {
                	dataStream.close();
                }
//            } else {
//                /*
//                 * fill first buffer from stream
//                 */
//                byte[] buffer = new byte[UPLOAD_BUFFER_SIZE];
//                ByteArrayInputStream dataStream = new ByteArrayInputStream(data);
//                int read = fillBuffer(dataStream, buffer);
//            	
//                /*
//                 * upload in multipart chunks to provide the correct content length
//                 */
//                String uploadID = initiateMultipartUpload(directoryPrefix + Storage.DELIMITER + ObjId);
//                boolean completed = false;
//                try {
//                    List<PartETag> partETags = new ArrayList<PartETag>();
//                    int partNumber = 1;
//                    do {
//                        ByteArrayInputStream bais = null;
//                        try {
//                            bais = new ByteArrayInputStream(buffer, 0, read);
//                            UploadPartRequest request = new UploadPartRequest().withBucketName(Config.getS3BucketName()).withKey(directoryPrefix + Storage.DELIMITER + ObjId).withUploadId(uploadID)
//                                .withInputStream(bais).withPartSize(read).withPartNumber(partNumber++);
//                            partETags.add(S3ClientFactory.s3Client.uploadPart(request).getPartETag());
//                        } finally {
//                        	bais.close();
//                        }
//                    } while (-1 != (read = fillBuffer(dataStream, buffer)));
//                    /*
//                     * complete upload
//                     */
//                    S3ClientFactory.s3Client.completeMultipartUpload(new CompleteMultipartUploadRequest(Config.getS3BucketName(), directoryPrefix + Storage.DELIMITER + ObjId, uploadID, partETags));
//                    completed = true;
//                } finally {
//                    if (false == completed) {
//                        try {
//                        	S3ClientFactory.s3Client.abortMultipartUpload(new AbortMultipartUploadRequest(Config.getS3BucketName(), directoryPrefix + Storage.DELIMITER + ObjId, uploadID));
//                        } catch (AmazonClientException e) {
//                        	Log.logError("Error aborting multipart upload");
//                        	Log.logError(e);
//                        }
//                    } else {
//                    	uploadSucess = true;
//                    }
//                }
//            }
        } catch (Exception e) {
        	logger.error("Error while uploading file to S3 store", e);
        }
        
        return uploadSucess;
    }
    
    public static EncryptedObj getEncrObj(String directoryPrefix, String ObjId) {
        try {
            InputStream in = S3ClientFactory.s3Client.getObject(Config.getS3BucketName(), directoryPrefix + Storage.DELIMITER + ObjId).getObjectContent();
            EncryptedObj obj = new EncryptedObj(in);
            in.close();
            return (obj);
        } catch (FileNotFoundException e) {
            logger.error("File \"" + directoryPrefix + Storage.DELIMITER + ObjId + "\" not found in bucket \"" + Config.getS3BucketName() + "\"");
            EncryptedObj obj = new EncryptedObj();
            obj.ItemID = "not found";
            return (obj);
        } catch (Exception e) {
        	logger.error("Error getEncrObj", e);
            return (null);
        }
    }
    
    public static InputStream getObjectStream(String directoryPrefix, String ObjId) {
        try {
            InputStream in = S3ClientFactory.s3Client.getObject(Config.getS3BucketName(), directoryPrefix + Storage.DELIMITER + ObjId).getObjectContent();
            return (in);
        } catch (Exception e) {
        	logger.error("Error getEncrObj", e);
            return (null);
        }
    }
    
    public static InputStream getObjectStream(String location) {
        try {
            InputStream in = S3ClientFactory.s3Client.getObject(Config.getS3BucketName(), location).getObjectContent();
            return (in);
        } catch (Exception e) {
            logger.error("Error getEncrObj", e);
            return (null);
        }
    }
    
    public static void deleteObj(String location) {
    	S3ClientFactory.s3Client.deleteObject(Config.getS3BucketName(), location);
    }
    
    /**
     * Initiates a new multipart upload for a file with the supplied key.
     *
     * @param key The (full) key for the new file; no additional prefix will be prepended implicitly
     * @return The upload ID for the multipart upload
     * @throws OXException
     */
    private static String initiateMultipartUpload(String key) throws AmazonClientException {
    	return S3ClientFactory.s3Client.initiateMultipartUpload(new InitiateMultipartUploadRequest(Config.getS3BucketName(), key)).getUploadId();
    }
    
    /**
     * Reads up as many bytes as possible from the supplied input stream into the buffer. Repeated reads are made until either the buffer
     * is filled completely, or the end of the stream is reached.
     *
     * @param inputStream the stream to read from
     * @param buffer The buffer to fill
     * @return The number of bytes read into the buffer, or <code>-1</code> if there is no more data because the end of the stream has
     *         been reached.
     * @throws IOException
     */
    private static int fillBuffer(InputStream inputStream, byte[] buffer) throws IOException {
        int offset = 0;
        boolean eof = false;
        do {
            int read = inputStream.read(buffer, offset, buffer.length - offset);
            if (-1 == read) {
                eof = true;
            } else {
                offset += read;
            }
        } while (offset < buffer.length && false == eof);
        return eof && 0 == offset ? -1 : offset;
    }
	
}
