package com.openexchange.guard.server.connectionPooling;

import java.io.IOException;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.TimeUnit;
import org.apache.http.HttpHost;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.conn.routing.HttpRoute;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.database.IdleConnectionMonitor;


/**
 * Provides a pool of HTTP connections
 */
class ConnectionPool {

    private final Logger logger = LoggerFactory.getLogger(ConnectionPool.class);
    private CloseableHttpClient httpClient;
    private PoolingHttpClientConnectionManager cm;
    private boolean resetting = false;
    private IdleConnectionMonitor monitor;
    private final CookieHandlingMode cookieHandlingMode;
    private final String name;

    /**
     * The cookie handling mode for a connection pool
     */
    public enum CookieHandlingMode{
        /**
         * Default cookie handling,
         */
        CookieHandlingEnabled,

        /**
         * Disable cookie handling
         */
        CookieHandligDissabled
    };

    /**
     * Constructor
     * @param name the name of the connection pool
     * @param cookieHandlingMode the cookie handling mode to use for the pool
     */
    ConnectionPool(String name, CookieHandlingMode cookieHandlingMode){
        this.name = name;
        this.cookieHandlingMode = cookieHandlingMode;
        initialize();
    }

    /**
     * Internal method to initialize the connection pool
     */
    private void initialize() {
        cm = new PoolingHttpClientConnectionManager(10, TimeUnit.MINUTES);
        cm.setMaxTotal(Config.max_total_connections);
        cm.setDefaultMaxPerRoute(Config.max_http_connections);
        HttpHost ox = new HttpHost(Config.OXBackend, Config.ox_backend_port);
        cm.setMaxPerRoute(new HttpRoute(ox), Config.max_http_connections);
        httpClient = buildHttpClient(cookieHandlingMode);
        monitor = new IdleConnectionMonitor(cm, name);
        monitor.start();
        try {
            monitor.join(1000);
        } catch (InterruptedException e) {
            logger.info("HTTP Monitor thread shutdown");
        }
    }

    /**
     * Internal method to build the HttpClient-based connection pool
     * @param cookieHandlingMode the cookie handling mode
     * @return A HTTClient representing the connection pool
     */
    private CloseableHttpClient buildHttpClient(CookieHandlingMode cookieHandlingMode) {

        RequestConfig requestConfig = RequestConfig.custom().
                                      setStaleConnectionCheckEnabled(true).
                                      setConnectTimeout(Config.connectionTimeout).
                                      setSocketTimeout(Config.connectionTimeout).
                                      build();

        return cookieHandlingMode == CookieHandlingMode.CookieHandligDissabled ?
            HttpClients.custom().setDefaultRequestConfig(requestConfig).
                                 disableCookieManagement().
                                 disableAutomaticRetries().
                                 setConnectionManager(cm).build() :
            HttpClients.custom().setDefaultRequestConfig(requestConfig).
                                 disableAutomaticRetries().
                                 setConnectionManager(cm).build();
    }

    /**
     * Closes the connection pool
     */
    public void close() {
        try {
            httpClient.close();
            monitor.shutdown();
            cm.close();
            cm.shutdown();
        } catch (IOException e) {
            logger.error("Error closing httpClient", e);
        }
    }

    /**
     * Resets the connection pool
     */
    public void reset() {
        try {
            if (resetting) {
                return;
            }
            logger.debug("Resetting connection pool");
            resetting = true;
            close();
            initialize();
            Timer resetDelay = new Timer();
            resetDelay.schedule(new TimerTask() {
                @Override
                public void run() {
                    resetting = false;
                    logger.debug("end reset delay");
                }
            }, 30000);
        } catch (Exception ex) {
            logger.error("Error resetting connection pool ", ex);
        }
    }

    /**
     * Gets a connection from the pool
     * @return
     */
    public CloseableHttpClient getHttpClient() {
        return this.httpClient;
    }


    /**
     * Gets the connection manager
     * @return
     */
    public PoolingHttpClientConnectionManager getConncetionManager() {
        return cm;
    }
}
