package com.openexchange.guard.database.remoteKeyCache;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import com.openexchange.guard.database.DbCommand;
import com.openexchange.guard.database.DbQuery;
import com.openexchange.guard.pgp.PGPPublicKeyRingExporter;
import com.openexchange.guard.pgp.PGPPublicKeyRingFactory;
import com.openexchange.guard.util.ListUtil;
import com.openexchange.guard.validator.EmailValidator;


public class RestDbRemoteKeyCacheStorage implements RemoteKeyCacheStorage {

    private final static String VALUES_LIST = "{@VALUES}";

    @Override
    public int deleteOld(int days) throws Exception {
        DbQuery db = new DbQuery();
        try {
            //Getting the old keys to delete
            DbCommand command = new DbCommand(RemoteKeyCacheSql.SELECT_OLD_STMT);
            command.addVariables(days);
            db.readOG(command);
            ArrayList<Long> deletingIds = new ArrayList<Long>();
            while(db.next()) {
                deletingIds.add(db.rs.getLong("id"));
            }

            //Deleting the stuff
            if(deletingIds.size() > 0) {

                //Remove the key entries
                String sql = RemoteKeyCacheSql.DELETE_REMOTE_KEYS_STMT;
                command = new DbCommand(sql.replace(VALUES_LIST, ListUtil.listToDbValuesList(deletingIds.size())));
                for(long id : deletingIds) {
                    command.addVariables(id);
                }
                db.writeOxGuard(command);

                //Remove key data
                sql = RemoteKeyCacheSql.DELETE_REMOTE_KEY_CONTENT_STMT;
                command = new DbCommand(sql.replace(VALUES_LIST, ListUtil.listToDbValuesList(deletingIds.size())));
                for(long id : deletingIds) {
                    command.addVariables(id);
                }
                db.writeOxGuard(command);
            }
            return deletingIds.size();
        }
        finally {
            db.close();
        }
    }

    @Override
    public void insert(PGPPublicKeyRing keyRing) throws Exception {
        DbQuery db = new DbQuery();
        try{
            //Storing the key ring data
            DbCommand command = new DbCommand(RemoteKeyCacheSql.INSERT_RING_DATA_STMT);
            String keyRingAsc = PGPPublicKeyRingExporter.export(keyRing);
            command.addVariables(keyRing.getPublicKey().getKeyID());
            command.addVariables(keyRingAsc);
            command.addVariables(keyRingAsc);
            db.writeOxGuard(command);

            //Storing each identity related to the key ring
            Iterator <String> ids = keyRing.getPublicKey().getUserIDs();
            Long refid = keyRing.getPublicKey().getKeyID();
            while (ids.hasNext()) {
                String email = ids.next().trim();
                if (email.contains("<")) {
                    email = email.substring(email.indexOf("<") + 1);
                    if (email.endsWith(">")) {
                        email = email.substring(0, email.length()-1);
                    }
                }
                if (EmailValidator.validate(email)) {
                    Iterator <PGPPublicKey> keys = keyRing.getPublicKeys();
                    while (keys.hasNext()) {
                        PGPPublicKey key = keys.next();
                        if (!key.isRevoked()) {
                            //Storing identity
                            command = new DbCommand(RemoteKeyCacheSql.INSERT_RING_ID_STMT);
                            command.addVariables(key.getKeyID());
                            command.addVariables(email);
                            command.addVariables(refid);
                            command.addVariables(refid);
                            db.writeOxGuard(command);
                        }
                    }
                }
            }
        }
        finally {
            db.close();
        }
    }

    @Override
    public List<PGPPublicKeyRing> getByEmail(String email) throws Exception {
        DbQuery db = new DbQuery();
        ArrayList<PGPPublicKeyRing> ret = new ArrayList<PGPPublicKeyRing>();
        try {
            DbCommand command = new DbCommand(RemoteKeyCacheSql.SELECT_BY_EMAIL_STMT);
            command.addVariables(email);
            db.readOG(command);
            while(db.next()) {
                ret.add(PGPPublicKeyRingFactory.create(db.rs.getString("ring")));
            }
            return ret;
        }
        finally {
            db.close();
        }
    }
}
