package com.openexchange.guard.database.ogKeyTable;

import java.util.List;
import com.openexchange.guard.encr.GuardKeys;


/**
 * Provides access to stored PGP Keys
 * {@link OGKeyTableStorage}
 */
public interface OGKeyTableStorage {

    /**
     * Gets a list of keys for a given user
     * @param userId the user's id
     * @param contextId the context id
     * @return a list of OX Guard keys for the given user
     * @throws Exception due an error
     */
    public List<GuardKeys> getKeysForUser(int userId,int contextId) throws Exception;

    /**
     * Gets a specific user key
     * @param userId the user id
     * @param contextId the context id
     * @param keyId the id of the key to get
     * @return the key with the given key id, or null if no such key was found
     * @throws Exception due an error
     */
    public GuardKeys getKeyForUserById(int userId, int contextId, String keyid) throws Exception;

    /**
     * Gets a specific user key from a shard
     * @param userId the user id
     * @param contextId the context id
     * @param keyId the id of the key to get
     * @param shardId the sharding id
     * @return the key with the given key id, or null if no such key was found
     * @throws Exception due an error
     */
    public GuardKeys getKeyForUserById(int userId, int contextId, int shardId, String keyId) throws Exception;

    /**
     * Gets the key with the highest version for a user.
     * @param userId
     * @param contextId
     * @return The user's keys with the highest version.
     * @throws Exception due an error
     */
    public GuardKeys getHighestVersionKeyForUser(int userId, int contextId) throws Exception;

    /**
     * Gets the user's key which is marked as "current" for a specific email
     * @param userId the user id
     * @param contextId the context id
     * @param email the email to get the current key for
     * @return the key for the user which is marked as current, or null if no such key was found
     * @throws Exception due an error
     */
    public GuardKeys getCurrentKeyForUser(int userId, int contextId, String email) throws Exception;

    /**
     * Gets the user's key which is marked as current
     * @param userId the user id
     * @param contextId the context id
     * @return the key for the user
     * @throws Exception due an error
     */
    public GuardKeys getCurrentKeyForUser(int userId, int contextId) throws Exception;

    /**
     * Gets the user's key which is marked as "current" from a specific shard
     * @param userId the user id
     * @param contextId the context id
     * @param shardId the sharding id
     * @return the key for the user which is marked as current, or null if no such key was found
     * @throws Exception due an error
     */
    public GuardKeys getCurrentKeyForUser(int userId, int contextId, int shardId) throws Exception;

    /**
     * Gets a list of keys for a given context
     * @param contextId the context id
     * @return a list of OX Guard keys for the given context
     * @throws Exception due an error
     */
    public List<GuardKeys> getKeysForContext(int contextId) throws Exception;

    /**
     * Gets the guard keys for a given email in a context
     * @param email the email
     * @param contextid the context
     * @return the GuardKey for the given email and context
     * @throws Exception due an error
     */
    public GuardKeys getKeyForEmailAndContext(String email, int contextid) throws Exception;
    
    /**
     * Gets the guard keys for a given email in a context
     * @param email the email
     * @param contextid the context
     * @param the shard where the key is located
     * @return the GuardKey for the given email and context
     * @throws Exception due an error
     */
    public GuardKeys getKeyForEmailAndContext(String email, int contextid, int shard) throws Exception;

    /**
     * Returns the guard server's own PGP key
     * @return the guard server's PGP key
     * @throws Exception due an error
     */
    public GuardKeys getGuardDaemonKey() throws Exception;

    /**
     * Inserts a new GuardKey
     * @param key the key to insert into
     * @param setLastMod true to set lastMod to NOW(), false to set it to NULL
     * @throws Exception due an exception
     */
    public void insert(GuardKeys key, boolean setLastMod) throws Exception;
    
    /**
     * Inserts a new GuardKey
     * @param key the key to insert into
     * @param setLastMod true to set lastMod to NOW(), false to set it to NULL
     * @param shard if guest user.
     * @throws Exception due an exception
     */
    public void insert(GuardKeys key, boolean setLastMod, int shard) throws Exception;

    /**
     * Updates the public part of a Guard key
     * @param key the guard key to update the public part for
     * @throws Exception due an error
     */
    public void updatePublicKey(GuardKeys key) throws Exception;

    /**
     * Updates the public and private part of a Guard key
     * @param keys the guard key to update the public and private part for
     * @throws Exception due an error
     */
    public void updatePublicAndPrivateKey(GuardKeys keys) throws Exception;

    /**
     * Updates the PGPSecret, RSAPrivate and the Recovery
     * @param key the key containing the new password values
     * @param the recovery to set
     * @param setLastMod true to set lastMod to NOW(), false to set it to NULL
     * @throws Exception due an error
     */
    public void updatePassword(GuardKeys key, String recovery, boolean setLastMod) throws Exception;

    /**
     * Updates the Key's pin which can be used as additional security token when sending emails to guest users
     * @param userId the ID of the user to update the pin for
     * @param contextId the contextId
     * @param newPin the new pin to set, or NULL to delete the pin
     */
    public void updatePin(int userId, int contextId, String newPin) throws Exception;

    /**
     * Updates an existing Guard Key (Updates PGPSecret,PGPPublic,Recovery and Salt)
     * @param key the new key data (PGPSecret, PGPPublic)
     * @param recovery the new recovery
     * @param updateDuplicatedKeyId the id of the key to update
     * @throws Exception due an error
     */
    public void updateDuplicate(GuardKeys key, String recovery, Long updateDuplicatedKeyId) throws Exception;

    /**
     * Updates the question and answer for all key's of a given user
     * @param userId the user's id
     * @param contextId the context id
     * @param question the new question
     * @param answer the new answer
     * @throws Exception
     */
    public void updateQuestionForUser(int userId, int contextId, String question, String answer) throws Exception;

    /**
     * Updates the version of a key
     * @param key the key to update
     * @param newVersion the new version to set
     * @throws Exception due an error
     */
    public void updateKeyVersion(GuardKeys key, int newVersion) throws Exception;

    /**
     * Marks the key as "current"
     * @param key the key to mark as current
     */
    public void setCurrentKey(GuardKeys key) throws Exception;

    /**
     * Marks all keys of the user as _not_ current
     * @param userId the user's id
     * @param contextId the context id
     * @throws Exception due an exception
     */
    public void unsetCurrentFlag(int userId, int contextId) throws Exception;

    /**
     * Deletes a specific key
     * @param key the key to delete
     */
    public void delete(GuardKeys key) throws Exception;

    /**
     * Deletes all keys related to a given user
     * @param userId the user id
     * @param contextId the context id
     * @throws Exception due an error
     */
    public void deleteAllForUser(int userId, int contextId) throws Exception;

    /**
     * Deletes all keys related for a given context
     * @param contextId the context id
     * @throws Exception due an error
     */
    public void deleteAllForContext(int contextId) throws Exception;

    /**
     * Removes the recovery
     * @param key the key to remove the recovery from
     * @throws Exception due an error
     */
    public void deleteRecovery(GuardKeys key) throws Exception;

    /**
     * Checks if the OGKeyStorage exists (if the underlying tables are present)
     * @param contextId the contextId
     * @return true if the storage exists, false otherwise
     * @throws Exception due an error
     */
    public boolean exists(int contextId) throws Exception;

    /**
     * Checks if the OGKeyStorage exists in the guest shards (if the underlying tables are present)
     * @param the name of the shard database to test
     * @return true if the storage exists, false otherwise
     * @throws Exception due an error
     */
    public boolean existsInShard(String shardDatabaseName) throws Exception;

    /**
     * Sets the auto increment within a guest shard
     * @param shardDatabaseName the name of the shard database
     * @param value the new auto increment value
     * @throws Exception due an error
     */
    public void updateAutoIncrementInShard(String shardDatabaseName, int value) throws Exception;
}
