/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.guard.database;

import java.io.ByteArrayInputStream;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.database.ogEmail.OGEmail;
import com.openexchange.guard.database.ogEmail.OGEmailStorage;
import com.openexchange.guard.database.ogEmail.RestDbOGEmailStorage;
import com.openexchange.guard.database.ogKeyTable.OGKeyTableStorage;
import com.openexchange.guard.database.ogKeyTable.RestDbOGKeyTableStorage;
import com.openexchange.guard.database.ogPGPKeys.OGPGPKey;
import com.openexchange.guard.database.ogPGPKeys.OGPGPKeysStorage;
import com.openexchange.guard.database.ogPGPKeys.RestDbOGPGPKeysStorage;
import com.openexchange.guard.database.oxguardSession.OXGuardSession;
import com.openexchange.guard.database.oxguardSession.OXGuardSessionStorage;
import com.openexchange.guard.database.oxguardSession.RestDbOXGuardSessionStorage;
import com.openexchange.guard.database.pgpKeys.PGPKeys;
import com.openexchange.guard.database.pgpKeys.PGPKeysStorage;
import com.openexchange.guard.database.pgpKeys.RestDbPGPKeysStorage;
import com.openexchange.guard.encr.EncrLib;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.pgp.PgpKeys;
import com.openexchange.guard.util.Core;

/**
 * Class for access of mysql database for key and content ID retrieval
 *
 * @author greg
 */
public class Access {

    private final String mysqlIP = com.openexchange.guard.config.Config.getMysqlIP();// IP address of configdb

    private final String mysqlUsername = com.openexchange.guard.config.Config.getMysqlUsername();

    private final String mysqlPassword = com.openexchange.guard.config.Config.getMysqlPassword();

    private final String OGmysqlIP = com.openexchange.guard.config.Config.getMysqlIPOxGuard();

    private final String ctxUsername = ""; // Username for access ox database that contains context

    private final String ctxPassword = ""; // Password

    private final String url = ""; // URL of the mysql server containing the context data

    private final String db = ""; // Database name

    private String OGmysqlConnectionString = "";

    private String ConfigmysqlConnectionString = "";

	private static Logger logger = LoggerFactory.getLogger(Access.class);

    static {
        try {
            Class.forName("com.mysql.jdbc.Driver");
        } catch (ClassNotFoundException e) {
            logger.error("Could not load jdbc driver", e);
        }
    }

    public Access() {
        ConfigmysqlConnectionString = Config.getConfigDbConnectionString();
        OGmysqlConnectionString = "jdbc:mysql://" + OGmysqlIP + "/" + Config.dbSchemaBaseName + "?user=" + mysqlUsername + "&password=" + mysqlPassword;

    }

    public String getEmail(int userid, int cid) throws SQLException {

        String email = "";
        try {
            String command = "SELECT mail from user WHERE cid = ? AND id = ?";
            DbCommand com = new DbCommand(command);
            com.addVariables(cid);
            com.addVariables(userid);
            DbQuery db = new DbQuery();
            db.readFromDB(com, cid);
            if (db.next()) {
                email = db.rs.getString("mail");
            }
            db.close();
        } catch (Exception ex) {
            logger.error("Error reading e-mail " + cid + "/" + userid + " from db", ex);
        }
        return (email);
    }

    public static String dbformat(String data) {
        if (data == null) {
            return (data);
        }
        data = data.replace("'", "\'");
        data = data.replace("\u00ef\u00bf\u00bd", "\'");
        data = data.replace("\\", "\\\\");
        return (data);
    }


    public GuardKeys storeKeys(GuardKeys key, boolean userCreated) throws Exception {
        GuardKeys ret = null;

        //Getting the shard to use
        int shard = 0;
        if (key.contextid < 0) { // If guest, get shard number, then check tables exist
            shard = Sharding.getNextShard();
            if (shard == 0) {
                logger.error("Unable to get non-ox shard id");
                return null;
            }
            Sharding.checkTables(shard, false);
        } else {
            if (key.contextid > 0)
             {
                Sharding.checkTables(key.contextid, true); // Check these OG tables exist
            }
        }

        //Adding the key to the storage
        OGKeyTableStorage keyTableStorage = new RestDbOGKeyTableStorage();
        keyTableStorage.insert(key,userCreated, shard);
        //retrieve the created key in order to get the userid which was created while inserting
        ret = keyTableStorage.getKeyForEmailAndContext(key.email,key.contextid, shard);

        //Store key mappings
        if(ret.userid > 0) {
            //Adding the new key to the email mapping storage
            addEmail(ret.email, ret.userid, ret.contextid, shard);

            //Adding the new key to the PGP key mapping storage
            PgpKeys.addPublicKeyIndex(ret);
        }

        return ret;
    }

    public boolean addNewKey (GuardKeys newkey) throws Exception {
        //TODO: use a Service-Locator for resolving this
        OGKeyTableStorage keyTableStorage = new RestDbOGKeyTableStorage();

        //removing the "current" flag from existing keys,
        //because the new created key will be the "current" key by default
        keyTableStorage.unsetCurrentFlag(newkey.userid, newkey.contextid);
        //Marking the new key as the "current" key
        newkey.current = true;

        //Getting the key with the highest version
        GuardKeys highestKeys = keyTableStorage.getHighestVersionKeyForUser(newkey.userid, newkey.contextid);

        //Incrementing key version, if highest was found, and adding the new key to the storage
        newkey.version = highestKeys != null ? highestKeys.version + 1 : 0;
        keyTableStorage.insert(newkey,true /*set lastMod to NOW()*/);

        //Adding the new key to the public key index (lookup table)
        PgpKeys.addPublicKeyIndex(newkey);

        return true;

    }

    public void updateKeys(int id, int cid, GuardKeys key, String recovery, boolean reset) throws Exception {
        //TODO: Use a DI Container to resolve this
        OGKeyTableStorage ogKeyTableStorage = new RestDbOGKeyTableStorage();
        ogKeyTableStorage.updatePassword(key, recovery, !reset /*setLastMod*/);
    }

    /**
     * Update PGP Key record updating PGP Secret key and recovery on duplicate
     * @param id
     * @param cid
     * @param key
     * @param recovery
     * @throws Exception
     */
    public void updateDuplPGPKey (int id, int cid, GuardKeys key, String recovery, Long dupl) throws Exception {
        //TODO: Use a service locator for resolving
        OGKeyTableStorage ogKeyTableStorage = new RestDbOGKeyTableStorage();
        ogKeyTableStorage.updateDuplicate(key, recovery, dupl);
    }

    public void updatePGPKeys (int id, int cid, GuardKeys key) throws Exception {
        //TODO: Use a service locator for resovling this
        OGKeyTableStorage ogKeyTableStorage = new RestDbOGKeyTableStorage();
        ogKeyTableStorage.updatePublicAndPrivateKey(key);
        PgpKeys.addPublicKeyIndex(key);
    }

    public static void updatePin(int id, int cid, String pin) throws Exception {
        //TODO: Use a service locator for resolving this
        OGKeyTableStorage ogKeyTableStorage = new RestDbOGKeyTableStorage();
        ogKeyTableStorage.updatePin(id,cid,pin);
    }

    public GuardKeys getKeys(int id, int cid) throws Exception {
        int shard = 0;
        if (cid < 0) {
            shard = Sharding.getShard(id, cid);
        }
        return (getKeys(id, cid, shard, null, true));
    }

    public GuardKeys getKeys(int id, int cid, long keyid, boolean current) throws Exception {
    	int shard = 0;
        if (cid < 0) {
            shard = Sharding.getShard(id, cid);
        }
        return (getKeys(id, cid, shard, Long.toString(keyid), current));
    }

    public GuardKeys getKeys (int id, int cid, int shard) throws Exception {
    	return (getKeys(id, cid, shard, null, true));
    }

    /**
     * Retrieve keys from Mater KeyTable
     *
     * @param id member ID
     * @return
     * @throws Exception
     */
    public GuardKeys getKeys(int id, int cid, int shard, String keyid, boolean current) throws Exception {
        //TODO: use a service-locator for resolving this
        OGKeyTableStorage ogKeyTableStorage = new RestDbOGKeyTableStorage();
        if(current) {
            return ogKeyTableStorage.getCurrentKeyForUser(id, cid, shard);
        }
        return ogKeyTableStorage.getKeyForUserById(id, cid, shard, keyid);
    }

    /**
     * Retrieve keys from Mater KeyTable
     *
     * @param id member ID
     * @return
     * @throws Exception
     */
    public GuardKeys getOldGuardKeys(int id, int cid) throws Exception {
        GuardKeys key = null;
        String command = "SELECT PGPSecret, PGPPublic, RSAPrivate, RSAPublic, Salt, misc, settings, email, lastMod, question, answer, cid, id, keyid, version, current, lang, Recovery FROM og_KeyTable WHERE id = ? AND cid = ? AND version < 1";  //Version may be 0 or -1 for Guard key (-1 during upgrade)
        DbCommand com = new DbCommand(command);
        com.addVariables(id);
        com.addVariables(cid);
        DbQuery db = new DbQuery();
        int shard = 0;
        if (cid < 0) {
            shard = Sharding.getShard(id, cid);
        }
        if (cid >= 0) {
            if (id < 0) {// minus 1 is the recovery password, which is in OG
                db.readOG(com);
            } else {
                db.readFromDB(com, cid);
            }
        } else {
            db.readFromShard(com, shard);
        }
        try {
            if (db.next()) {
                key = getKeyFromRS (db.rs);
            }
        } catch (Exception ex) {
            logger.error("Error while iterating result set for keys", ex);

        }
        return (key);
    }

    /**
     * Create GuardKey from database result set
     * @param rs
     * @param id  ID of user, -1 if unknown
     * @param cid
     * @return
     * @throws InvalidKeySpecException
     * @throws NoSuchAlgorithmException
     * @throws Exception
     */
    private GuardKeys getKeyFromRS (OxResultSet rs) throws InvalidKeySpecException, NoSuchAlgorithmException, Exception {
        GuardKeys key = new GuardKeys();
        key.setPGPSecretFromString(rs.getString("PGPSecret"));
        key.setPGPKeyRingFromAsc(rs.getString("PGPPublic"));
        key.setEncodedPrivate(rs.getString("RSAPrivate"));
        key.setPublicKeyFrom64String(rs.getString("RSAPublic"));
        key.setSalt(rs.getString("Salt"));
        key.misc = rs.getString("misc");
        key.settings = rs.getJson("settings");
        key.email = rs.getString("email");
        key.lastup = rs.getDate("lastMod");
        key.question = rs.getString("question");
        key.answer = rs.getString("answer");
        key.contextid = rs.getInt("cid");
        key.userid = rs.getInt("id");
        key.keyid = rs.getLong("keyid");
        key.version = rs.getInt("version");
        key.current = rs.getBit("current");
        key.language = rs.getString("lang");
        key.recovery = rs.getString("Recovery");
        key.recoveryAvail = key.recovery == null ? false : !key.recovery.equals("");
        if (key.recovery != null) {
            if (key.recovery.equals("-1") || (key.lastup == null)) {
                key.passwordNeeded = true;
            }
        }
        return (key);
    }

    public GuardKeys getKeysFromEmail(String email) throws Exception {
        return (getKeysFromEmail(email, 0L));
    }
    /**
     * Retrieve keys from Mater KeyTable
     *
     * @param id member ID
     * @return
     * @throws Exception
     */
    public GuardKeys getKeysFromEmail(String email, Long keyid) throws Exception {
        GuardKeys ret = null;

        //TODO: Use a DI container to resolve this
        OGEmailStorage ogEmailStorage = new RestDbOGEmailStorage();
        OGEmail ogEmail = ogEmailStorage.getByEmail(email);

        if(ogEmail != null) {
            if (keyid != 0L) {
                ret = getKeys(ogEmail.getUserId(),
                              ogEmail.getContextId(),
                              ogEmail.getShardingId(),
                              Long.toString(keyid), false);
            }
            else {
                ret = getKeys(ogEmail.getUserId(),
                              ogEmail.getContextId(),
                              ogEmail.getShardingId());
            }
        }
        return ret;
    }

    /**
     * Store Item into item table
     *
     * @param ItemID
     * @param owner
     * @param exp
     * @param type
     * @param XML
     * @throws Exception
     */
    public void storeItemID(String ItemID, int recip, int recip_cid, int owner, int owner_cid, long exp, int type, String XML, String salt) throws Exception {
        // Store item. If multiple recips within the same context database, then there will be duplicates we can ignore
        if (XML.length() > 3072) {
            XML = XML.substring(0, 3071);
        }
        String command = "INSERT INTO og_encrypted_items (Id, Owner, Expiration, Type, XML, Salt, Owner_cid) VALUES (?, ?, ?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE Type = Type";
        DbQuery db = new DbQuery();
        DbCommand com = new DbCommand(command);
        com.addVariables(ItemID);
        com.addVariables(owner);
        com.addVariables(exp);
        com.addVariables(type);
        com.addVariables(XML);
        com.addVariables(salt);
        com.addVariables(owner_cid);
        db.write(com, recip, recip_cid);
        db.close();

    }

    public void storeContentKey(String itemId, int userid, int cid, String contentKey) throws Exception {

        String command = "INSERT INTO og_content_keys (ItemId, UserId, CID, ContentKey, Status, ID) VALUES (?, ?, ?, ?, 0, NULL);";
        DbCommand com = new DbCommand(command);
        com.addVariables(itemId);
        com.addVariables(userid);
        com.addVariables(cid);
        com.addVariables(contentKey);
        DbQuery db = new DbQuery();
        db.write(com, userid, cid);
    }

    private String convertDate(java.util.Date dt) {
        if (dt == null) {
            return (null);
        }
        java.text.SimpleDateFormat sdf = new java.text.SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        String currentTime = sdf.format(dt);
        return (currentTime);
    }

    public Boolean itemExists(String ItemId, int user, int cid) throws Exception {
        String command = "SELECT Id FROM og_encrypted_items WHERE og_encrypted_items.ID = ? AND Owner = ? AND Owner_cid = ?;";
        DbQuery db = new DbQuery();
        DbCommand com = new DbCommand(command);
        com.addVariables(ItemId);
        com.addVariables(user);
        com.addVariables(cid);
        db.read(com, user, cid);
        if (db.hasResults()) {
            return (true);
        }
        return (false);
    }

    public void updateAccess(String ItemID, int user, int cid) {
        String command = "UPDATE og_content_keys set Status = Status + 1 WHERE ItemID = ? AND UserId = ? AND CID = ?";
        try {
            DbQuery db = new DbQuery();
            DbCommand com = new DbCommand(command);
            com.addVariables(ItemID);
            com.addVariables(user);
            com.addVariables(cid);
            db.write(com, user, cid);

        } catch (Exception e) {
            logger.error("Error updating status", e);
        }
    }

    public void getCount(String ItemID, int userid, int cid) throws Exception {
        String xml = "";
        String command = "SELECT XML FROM og_encrypted_items WHERE ID = ? AND Owner = ? AND Owner_cid = ?;";
        DbQuery db = new DbQuery();
        DbCommand com = new DbCommand(command);
        com.addVariables(ItemID);
        com.addVariables(userid);
        com.addVariables(cid);
        db.read(com, userid, cid);
        if (db.next()) {
            xml = db.rs.getString("XML");
        }
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        try {
            DocumentBuilder builder = factory.newDocumentBuilder();
            org.w3c.dom.Document doc = builder.parse(new InputSource(new ByteArrayInputStream(xml.getBytes("utf-8"))));
            doc.getDocumentElement().normalize();
            NodeList users = doc.getElementsByTagName("u");
            StringBuilder output = new StringBuilder();
            output.append("[");
            for (int i = 0; i < users.getLength(); i++) {
                if (i > 0) {
                    output.append(", ");
                }
                Element user = (Element) users.item(i);
                String email = user.getElementsByTagName("e").item(0).getTextContent();
                int uid = Integer.parseInt(user.getElementsByTagName("i").item(0).getTextContent());
                int ucid = Integer.parseInt(user.getElementsByTagName("c").item(0).getTextContent());
                output.append("{\"Email\" : \"" + email + "\", \"Count\" : \"" + getCountbyUser(ItemID, uid, ucid) + "\"}");
            }
            output.append("]");
        } catch (Exception ex) {
            logger.error("Error while getting count of encrypted items", ex);
        }

    }

    private int getCountbyUser(String itemid, int id, int cid) {
        int count = 0;
        try {
            String command = "SELECT Status FROM og_content_keys WHERE ItemId = ? AND UserID = ? AND CID = ?";
            DbCommand com = new DbCommand(command);
            com.addVariables(itemid);
            com.addVariables(id);
            com.addVariables(cid);
            DbQuery db = new DbQuery();
            db.read(com, id, cid);
            if (db.next()) {
                count = db.rs.getInt("Status");
            }
        } catch (Exception ex) {
            logger.error("Error while getting count of encrypted items for user " + id, ex);
        }
        return (count);
    }

    public RetrievedItem getItemID(String ItemId, int user, int cid) throws Exception {
        String command = "SELECT ContentKey, Expiration, Owner, Salt, Type, XML FROM og_encrypted_items INNER JOIN og_content_keys ON og_encrypted_items.Id = og_content_keys.ItemId WHERE og_encrypted_items.ID = ? AND UserId = ? AND CID = ?;";
        DbCommand com = new DbCommand(command);
        com.addVariables(ItemId);
        com.addVariables(user);
        com.addVariables(cid);
        DbQuery db = new DbQuery();
        db.read(com, user, cid);
        if (db.next()) {
            OxResultSet rs = db.rs;
            RetrievedItem item = new RetrievedItem();
            item.contentKey = rs.getString("ContentKey");
            item.expiration = rs.getLong("Expiration");
            item.owner = rs.getInt("Owner");
            item.salt = rs.getString("Salt");
            item.Type = rs.getInt("Type");
            item.XML = rs.getString("XML");
            return (item);

        }
        return (null);
    }

    /**
     * Insert new token to oxGuard_sessions for encrypting passwords
     *
     * @param sessionId the session ID to create a new token for
     * @param userId the user's id
     * @param userid the user's id
     * @return the new created token
     * @throws Exception due an error
     */
    public String newToken(String sessionId, int userId, int contextId) throws Exception {
        //Create a new token
        String token = EncrLib.genRandom64Key();

        //Insert or update the token
        //TODO: Use a service locator / DI Container for resolving
        OXGuardSessionStorage guardSessionStorage = new RestDbOXGuardSessionStorage();
        guardSessionStorage.InsertOrUpdate(sessionId, token, userId, contextId);

        return token;
    }

    /**
     * Delete token from sessions for logout
     *
     * @param sessionID
     * @throws Exception
     */
    public void deleteToken(String sessionID) throws Exception {
        if (sessionID != null) {
            //TODO: use a DI-Container for resolving this
            OXGuardSessionStorage guardSessionStorage = new RestDbOXGuardSessionStorage();
            guardSessionStorage.deleteById(sessionID);
            guardSessionStorage.deleteOldSessions(1);
        }
    }

    /**
     * Get the token associated with the sessionID
     *
     * @param sessionID the session to get the token for
     * @return the token related to the given session or null if no token was found for the sessions
     */
    public String getToken(String sessionId) {
        try {

            //TODO: using a DI-Container to resolve this
            OXGuardSessionStorage guardSessionStorage = new RestDbOXGuardSessionStorage();

            OXGuardSession session = guardSessionStorage.getById(sessionId);
            if(session != null) {
                return session.getToken();
            }
        } catch (Exception e) {
            logger.error("Error while retrieving token from session " + sessionId, e);
        }
        return (null);
    }


    // Lookup the public key and userid for a given email address. Returns null if not found

    /**
     * Get number of times item has been opened. Returns -1 if not found
     *
     * @param userid
     * @param cid
     * @param itemid
     * @return
     */
    public int getCount(int userid, int cid, String itemid) {
        int response = -1;
        try {
            String command = "SELECT Status FROM og_content_keys WHERE ItemID = ? AND UserID = ? AND CID = ?";
            DbQuery db = new DbQuery();
            DbCommand com = new DbCommand(command);
            com.addVariables(itemid);
            com.addVariables(userid);
            com.addVariables(cid);
            db.read(com, userid, cid);
            if (db.next()) {
                response = db.rs.getInt("Status");
            }
        } catch (Exception ex) {
            logger.error("Error while getting count of content keys", ex);
        }
        return (response);
    }

    /**
     * Retract an item by setting expiration to negative number Salt is required for extra security check
     */

    public boolean retract(int userid, int cid, String itemID, String salt, boolean delete) {
        try {
            DbQuery db = new DbQuery();
            String command = "SELECT COUNT(Id) AS count FROM og_encrypted_items WHERE  og_encrypted_items.Id = ? AND Salt = ?";
            // Salt must be same to verify authorized sender
            DbCommand com = new DbCommand(command);
            com.addVariables(itemID);
            com.addVariables(salt);
            db.read(com, userid, cid);
            if (db.next()) {
            	if (db.rs.getInt("count") < 1) {
            		db.close();
            		return(false); // no items to retract
            	}
            }
            // Delete the content key if so configured, otherwise expire the item
            command = "update og_encrypted_items SET Expiration = IF (Expiration = 0, -1, -Expiration) WHERE ID = ? AND Expiration >= 0 AND Salt = ?";

            if (delete) {
            	command = "update og_content_keys SET ContentKey = 'retracted' WHERE ItemId = ?";
            }
            com = new DbCommand(command);
            com.addVariables(itemID);
            if (!delete) {
                com.addVariables(salt);
            }
            db.write(com, userid, cid);
            command = "update og_content_keys SET Status = IF (Status = 0, -1, -Status -1) WHERE ItemId = ?";
            com = new DbCommand(command);
            com.addVariables(itemID);
            db.write(com, userid, cid);
            return (true);
        } catch (Exception ex) {
            logger.error("Error while retracting an encrypted item", ex);
            return (false);
        }
    }

    /**
     * Get XML data from an item. Returns null if not found. User must be owner
     *
     * @param userid
     * @param cid
     * @param itemid
     * @return
     */
    public String getXML(int userid, int cid, String itemid) {
        String response = null;
        try {
            String command = "SELECT XML FROM og_encrypted_items WHERE Id = ? AND Owner = ? AND Owner_cid = ?";
            DbQuery db = new DbQuery();
            DbCommand com = new DbCommand(command);
            com.addVariables(itemid);
            com.addVariables(userid);
            com.addVariables(cid);
            db.read(com, userid, cid);
            if (db.next()) {
                response = db.rs.getString("XML");
            }
        } catch (Exception ex) {
            logger.error("Error while getting XML for encrypted item", ex);
        }
        return (response);
    }

    public boolean storeQuestion(int id, int cid, String question, String answer) {
        EncrLib encr = new EncrLib();
        String encranswer = null;
        if (answer != null) {
            encranswer = encr.encryptAES(answer, Config.rpass, question, Config.AESKeyLen);
        }
        try {
            OGKeyTableStorage ogKeyTableStorage = new RestDbOGKeyTableStorage();
            ogKeyTableStorage.updateQuestionForUser(id, cid, question, encranswer);
        } catch (Exception e) {
            logger.error("Error while storing reminder question", e);
            return(false);
        }
        return(true);
    }

    /**
     * Search for public keys for email
     * @param email
     * @return
     * @throws Exception due an error
     */
    public ArrayList<PGPPublicKeyRing> getPublicKeyRing(String email) throws Exception {
            ArrayList<PGPPublicKeyRing> ret = new ArrayList<PGPPublicKeyRing>();

            //TODO: Use a service locator for resolving this
            PGPKeysStorage pgpKeysStorage = new RestDbPGPKeysStorage();
            List<PGPKeys> internalKeyMappings = pgpKeysStorage.getInternalByEmail(email);
            for(PGPKeys internalMapping : internalKeyMappings) {
                if(internalMapping.isLocal()) {
                    GuardKeys key = getKeysFromEmail(email);
                    ret.add(key.pubring);

                }
            }
            return ret;
    }

    /**
     * Get the public keyring associated with the id.
     * @param req_id
     * @param userid
     * @param cid
     * @return
     */
    public PGPPublicKeyRing getPublicKeyRingById(long req_id, int userid, int cid) {
        try {
            PGPKeysStorage pgpKeysStorage = new RestDbPGPKeysStorage();
            PGPKeys keyMapping = pgpKeysStorage.getById(req_id);
            if(keyMapping != null) {
                if(keyMapping.isLocal()) {
                    GuardKeys key = getKeysFromEmail(keyMapping.getEmail(), keyMapping.getKeyId());
                    return key.pubring;
                }
            }

            String hexid = Core.PGPKeyId(req_id);
            return (getUploadedPublicKeyRingById (hexid, userid, cid));
        } catch (Exception ex) {
            logger.error("Error while retrieving public key ring for " + req_id, ex);
            return (null);
        }
    }

    /**
     * Get the public keyring associated with the id.
     * @param req_id
     * @param userid
     * @param cid
     * @return
     */
    public PGPPublicKeyRing getPublicKeyRingById(String hexid, int userid, int cid) {
        try {
            PGPKeysStorage pgpKeysStorage = new RestDbPGPKeysStorage();
            PGPKeys keyMapping = pgpKeysStorage.getByHexId(hexid);
            if(keyMapping != null) {
                if(keyMapping.isLocal()) {
                    GuardKeys key = getKeysFromEmail(keyMapping.getEmail(), keyMapping.getKeyId());
                    return key.pubring;
                }
            }
            return (getUploadedPublicKeyRingById (hexid, userid, cid));

        } catch (Exception ex) {
            logger.error("Error while retrieving public key ring for " + hexid, ex);
            return (null);
        }
    }

    private PGPPublicKeyRing getUploadedPublicKeyRingById (String req_id, int userid, int cid) throws Exception {
        //TODO: Use a DI-Container or Service locator to resolve this
        if (cid == 0) return (null);
        OGPGPKeysStorage ogpgpKeysStorage = new RestDbOGPGPKeysStorage();
        List<OGPGPKey> pgpKeys = ogpgpKeysStorage.getForUserByIds(userid, cid, Arrays.asList(new String[] {req_id}));
        if(pgpKeys.size() > 0) {
            GuardKeys key = new GuardKeys();
            key.setPGPKeyRingFromAsc(pgpKeys.get(0).getPublicPGPAscData());
            return key.pubring;
        }
        return null;
    }

    public static void addEmail(String email, int id, int cid, int shard) throws Exception {
        OGEmailStorage ogEmailStorage = new RestDbOGEmailStorage();
        ogEmailStorage.insertOrUpdate(email, cid, id, shard);
    }
}
