/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.cron;

import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.session.GuardSessionService;

/**
 * {@link CleanAuthenticationTokenTask} invalidates authentication tokens which are older than a specific amount of time
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.8.2
 */
public class CleanAuthenticationTokenTask implements Runnable {

    private static org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(CleanAuthenticationTokenTask.class);
    private final long maxTokenLifeTime;

    /**
     * Initializes a new {@link CleanAuthenticationTokenTask}.
     * 
     * @param maxTokenLifeTime The amount of time after an authentication token should be cleaned up from the database
     */
    public CleanAuthenticationTokenTask(long maxTokenLifeTime) {
        this.maxTokenLifeTime = maxTokenLifeTime;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Runnable#run()
     */
    @Override
    public void run() {
        try {
            GuardSessionService guardSessionService = Services.getService(GuardSessionService.class);
            logger.info("Deleting authentication tokens which are older than {} ms.", maxTokenLifeTime);
            int tokensDeleted = guardSessionService.deleteTokens(maxTokenLifeTime);
            logger.info("Deleted {} expired authentication tokens which were older than {} ms.", tokensDeleted, maxTokenLifeTime);
        } catch (Exception e) {
            logger.error("Error while cleaning expired authentication tokens.", e);
        }
    }
}
