/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.milter.internal;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.channels.WritableByteChannel;
import org.slf4j.Logger;
import com.openexchange.exception.OXException;
import com.openexchange.guard.milter.actions.encryptor.MilterEOBActionEncryptor;
import com.openexchange.guard.milter.actions.encryptor.MilterRcptActionEncryptor;
import com.openexchange.guard.milter.service.MilterHandlerService;
import com.openexchange.guard.milter.service.MilterResponse;
import com.openexchange.guard.milter.service.impl.ActionResponse;
import com.openexchange.guard.milter.service.impl.DataResponse;
import com.openexchange.guard.milter.service.impl.GuardMilterHandlerService;
import com.openexchange.guard.milter.service.impl.GuardMilterMail;
import com.openexchange.guard.milter.service.impl.NoResponse;
import com.openexchange.guard.milter.service.impl.StandardResponse;

/**
 * {@link ConnectionHandler}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.8.0
 */
public class ConnectionHandler {

    static Logger Log = org.slf4j.LoggerFactory.getLogger(ConnectionHandler.class);

    MilterHandlerService milter;

    Boolean aborted;

    public ConnectionHandler () {
        milter = new GuardMilterHandlerService();
        milter.addRcptAction(new MilterRcptActionEncryptor());
        milter.addEOBAction(new MilterEOBActionEncryptor());
        aborted = false;
    }

    public MilterResponse handle (WritableByteChannel channel, MilterPacket packet) throws IOException {

        MilterResponse resp = null;
        try {
            switch (packet.getCommand()) {
                case MilterConstants.COMMAND_CONNECT:
                    handleConnect(packet);
                    resp = new StandardResponse(MilterConstants.RESPONSE_CONTINUE);
                    break;
                case MilterConstants.COMMAND_HELO:
                    resp = new StandardResponse(MilterConstants.RESPONSE_CONTINUE);
                    break;
                case MilterConstants.COMMAND_OPTNEG:
                    return (handleOpt(channel));
                case MilterConstants.COMMAND_RCPT:
                    resp = milter.rcptTo(packet);
                    break;
                case MilterConstants.COMMAND_HEADER:
                    resp = milter.header(packet);
                    break;
                case MilterConstants.COMMAND_MACRO:
                    resp = new NoResponse();
                    break;
                case MilterConstants.COMMAND_BODY:
                    resp = milter.body(packet);
                    break;
                case MilterConstants.COMMAND_QUIT:
                    return null;
                case MilterConstants.COMMAND_ABORT:
                    resp = new StandardResponse(MilterConstants.RESPONSE_CONTINUE);
                    milter.reset();
                    break;
                case MilterConstants.COMMAND_BODYEOB:
                    resp = milter.handleEOB (packet, channel);
                    break;
                default:
                    resp = new StandardResponse(MilterConstants.RESPONSE_CONTINUE);
                    break;
            }

        }  catch (OXException ex) {
            Log.error("Problem handling Milter ", ex);
            resp = new StandardResponse(MilterConstants.RESPONSE_CONTINUE);  // Fallback is to just continue
        }

        return resp;

    }

    private static void handleConnect (MilterPacket packet) {
        ByteBuffer data = packet.getData();
        String host = MilterUtils.getStringFromData(data);
        if (data.get() == MilterConstants.SMFIA_INET) {
            data.getShort();  //port
            String addr = MilterUtils.getStringFromData(data);
            Log.debug("Milter Connection from " + host + " IP: " + addr);
        }
    }

    private static MilterResponse handleOpt (WritableByteChannel channel) throws IOException {
        ByteBuffer optionData = ByteBuffer.wrap( new byte[] {
            0x00, 0x00, 0x00, 0x02, // version
            0x00, 0x00, 0x00, 0x00, // (byte) smfif, // SMFIF_
            0x00, 0x00, 0x00, 0x00, // (byte) smfip, // SMFIP_
        });

        return new DataResponse(MilterConstants.COMMAND_OPTNEG, optionData);
    }


}
