/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.guest.redirector;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import com.openexchange.exception.OXException;
import com.openexchange.guard.database.GuardDatabaseService;
import com.openexchange.guard.database.utils.GuardConnectionWrapper;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;

/**
 * {@link GuestRedirectorServiceImpl}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.0
 */
public class GuestRedirectorServiceImpl implements GuestRedirectorService {

    private final GuardDatabaseService guardDatabaseService;

    public GuestRedirectorServiceImpl (GuardDatabaseService guardDatabaseService) {
        this.guardDatabaseService = guardDatabaseService;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.guest.redirector.GuestRedirectorService#addRedirect(int, int, java.lang.String)
     */
    @Override
    public void addRedirect(int userId, int contextId, String redirect) throws OXException {
        GuardConnectionWrapper connectionWrapper = guardDatabaseService.getWritable(userId, contextId, 0);
        Connection connection = connectionWrapper.getConnection();
        try(PreparedStatement stmt = connection.prepareStatement(GuestRedirectorSQL.INSERT_REDIRECT)) {
            stmt.setInt(1, userId);
            stmt.setInt(2, contextId);
            stmt.setString(3, redirect);
            stmt.execute();
        } catch (SQLException e) {
            throw GuardCoreExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            guardDatabaseService.backWritable(connectionWrapper);
        }

    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.guest.redirector.GuestRedirectorService#getRedirect(int, int)
     */
    @Override
    public String getRedirect(int userId, int contextId) throws OXException {
        GuardConnectionWrapper connectionWrapper = guardDatabaseService.getReadOnly(userId, contextId, 0);
        Connection connection = connectionWrapper.getConnection();
        String url = null;
        try(PreparedStatement stmt = connection.prepareStatement(GuestRedirectorSQL.SELECT_REDIRECT)) {
            stmt.setInt(1, userId);
            stmt.setInt(2, contextId);
            ResultSet rs = stmt.executeQuery();
            if (rs.next()) {
                url = rs.getString(1);
            }
            rs.close();
        } catch (SQLException e) {
            throw GuardCoreExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            guardDatabaseService.backReadOnly(connectionWrapper);
        }
        return url;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.guest.redirector.GuestRedirectorService#deleteRedirect(int, int)
     */
    @Override
    public void deleteRedirect(int userId, int contextId) throws OXException {
        GuardConnectionWrapper connectionWrapper = guardDatabaseService.getWritable(userId, contextId, 0);
        Connection connection = connectionWrapper.getConnection();
        try(PreparedStatement stmt = connection.prepareStatement(GuestRedirectorSQL.DELETE_REDIRECT)) {
            stmt.setInt(1, userId);
            stmt.setInt(2, contextId);
            stmt.execute();
        } catch (SQLException e) {
            throw GuardCoreExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            guardDatabaseService.backWritable(connectionWrapper);
        }

    }

}
