/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 * © 2017 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Greg Hill <greg.hill@open-xchange.com>
 */

/// <reference path="../../steps.d.ts" />

Feature('Smime: Settings');

Before(async function (users) {
    await users.create().then(async function (user) {
        await user.hasCapability('smime');
        await user.doesntHaveCapability('guard');
    });
    
});

After(async function (users) {
    await users.removeAll();
});

Scenario('Smime settings correct and key avail', async function (I, users) {

    const o = {
        user: users[0]
    };
    const userdata = o.user.userdata;

    // Log in to settings
    I.login('app=io.ox/mail', o);
    await I.setupSmimeKey(o.user);

    // Next, log in to settings
    await I.goToSettings();

    I.see('Default to send encrypted when composing email');
    I.see('Default adding signature to outgoing mails');

    I.see('Change password');

    I.see('Your keys');
    I.click('Your keys');

    I.waitForElement('.keytable');

    I.see(userdata.email1);

    I.see('test.ox');  // certifier

    I.click('Done');

});

Scenario('Smime change password', async function (I, users) {

    const o = {
        user: users[0]
    };
    const userdata = o.user.userdata;

    // Log in to settings
    I.login('app=io.ox/mail', o);
    await I.setupSmimeKey(o.user);

    // Next, log in to settings
    await I.goToSettings();

    I.see('Change password');

    I.click('Change password');

    I.waitForVisible('#oldpass');

    I.fillField('#oldpass', userdata.password);

    I.fillField('#newpass1', userdata.password + '2');

    I.fillField('#newpass2', userdata.password);

    I.see('Passwords not equal');

    I.pressKey('2');

    I.click('Change');

    I.waitForVisible('.io-ox-alert');

    I.see('Password changed', '.io-ox-alert');

    I.waitForVisible('.btn[name="yourKeys"]');
    I.click('.btn[name="yourKeys"]');
    I.waitForVisible('.keytable .fa-check');
    // Confirm only 1 key
    I.seeNumberOfVisibleElements('.keytable tr', 1);


});

Scenario('Smime Reset Password and Change', async function (I, users) {

    const o = {
        user: users[0]
    };
    const userdata = o.user.userdata;

    I.login('app=io.ox/mail', o);
    
    await I.setupSmimeKey(o.user);

    await I.logout();  // we have to login again so that we test smime login data shows reset avail

    I.login('app=io.ox/mail', o);

    // Next, log in to settings
    await I.goToSettings();

    // Reset the password
    I.see('Reset password');
    I.click('Reset password');
    I.waitForVisible('.modal-title');
    I.seeTextEquals('Reset Password', '.modal-title');  // Header
    I.click('.btn[data-action="reset"]');
    I.waitForInvisible('.modal-dialog');

    // Goto mail and wait for mail to arrive
    I.openApp('Mail');
    I.waitForVisible('.io-ox-mail-window .leftside');
    I.wait(1);
    I.click('#io-ox-refresh-icon');
    I.waitForVisible('.io-ox-mail-window .leftside ul li.unread');
    I.click('.io-ox-mail-window .leftside ul li.unread');

    //Pull the new temporary password from the email
    I.waitForVisible('.mail-detail-frame');
    I.switchTo('.mail-detail-frame');
    let newPass = await I.grabTextFrom('.bodyBox');
    I.switchTo();

    // Got back to settings and change the password
    await I.goToSettings();
    I.waitForVisible('#changePassword');
    I.click('#changePassword');
    I.waitForVisible('#oldpass');
    I.fillField('#oldpass', newPass);
    I.insertCryptPassword('#newpass1', userdata);
    I.insertCryptPassword('#newpass2', userdata);
    I.click('.btn[data-action="change"]');

    // Confirm good change of the temporary password
    I.waitForVisible('.io-ox-alert-success');
    I.seeTextEquals('Password changed successfully.', '.io-ox-alert-success');

    I.logout();

});

Scenario('No Smime key recognized in settings', async function (I, users) {

    const o = {
        user: users[0]
    };
    const userdata = o.user.userdata;

    // Log in to settings
    I.login(['app=io.ox/settings', 'folder=virtual/settings/oxguard'], o);
    I.waitForElement('.io-ox-guard-settings');
    I.waitForElement('.guard-start-button');

    // Click the setup button and verify wizard;
    I.click('.guard-start-button');

    I.waitForElement('.modal-title');
    I.see('Upload S/MIME Keys');
});

Scenario('I can upload an smime key', async function (I, users) {

    const o = {
        user: users[0]
    };
    const userdata = o.user.userdata;

    // Log in to settings
    I.login(['app=io.ox/settings', 'folder=virtual/settings/oxguard'], o);
    I.waitForElement('.io-ox-guard-settings');
    I.waitForElement('.guard-start-button');

    // Click the setup button and verify wizard;
    I.click('.guard-start-button');

    I.waitForElement('.modal-title');
    I.see('Upload S/MIME Keys');

    I.fillField('#pgppassword', 'secret');
    I.fillField('#guardpassword', 'secret');
    I.fillField('#guardpassword2', 'secret');

    // I.click('OK');

    I.waitForElement('input[type="file"]');
    await I.executeScript(function() {
        var input = $('#smimeKeyFileInput');
            input.on('change', function () {
                var files = this.files;
                if (files.length > 0) {
                    require(['oxguard/smime/uploader'], function (smimeUploader) {
                       smimeUploader.upload(files, $('#pgppassword').val(), $('#guardpassword').val()).done(function (data) {
                       });
                    });
                }
            });
    });

    I.attachFile('input[type="file"]', 'e2e/testFiles/smime.p12');

    I.wait(1);

    I.click('Cancel');  // we don't go through the file input dialog

    I.waitForVisible('.btn[name="yourKeys"]');

    I.see('Your keys');
    I.click('Your keys');

    I.waitForElement('.keytable');

    I.see('test@encr.us');

    I.see('test.ox');  // certifier

    I.click('Done');
    
})

Scenario('Create keys error from within compose', async function (I, users) {
    const o1 = {
        user: users[0]
    };

    const userdata = o1.user.userdata;

    I.login('app=io.ox/mail', o1);
    I.waitForVisible('.io-ox-mail-window .window-body .classic-toolbar');

    I.click('Compose');
    I.waitForVisible('.window-blocker.io-ox-busy');
    I.waitForInvisible('.window-blocker.io-ox-busy');
    I.waitForVisible('.io-ox-mail-compose textarea.plain-text,.io-ox-mail-compose .contenteditable-editor');

    I.click('.toggle-encryption');

    I.waitForVisible('.io-ox-alert');

    I.see('No S/MIME keys available');

    I.dontSee('.fa.encrypted');  // unlocked
});

function confirm(checked, location) {
    const { I } = inject();
    within(locate('.keytable tr').at(location), () => {
        if (checked) {
            I.seeElement('.fa-check');
        } else {
            I.seeElement('input.makecurrent');
        }
    })
}

Scenario('Select current smime key', async function (I, users) {

    const o = {
        user: users[0]
    };

    I.login('app=io.ox/mail', o);
    await I.setupSmimeKey(o.user);  // two keys
    await I.setupSmimeKey(o.user);

    // Next, log in to settings
    await I.goToSettings();

    // Open keys
    I.waitForVisible('.btn[name="yourKeys"]');
    I.click('.btn[name="yourKeys"]');
    I.waitForVisible('.keytable .fa-check');
    // Confirm 2 rows
    I.seeNumberOfVisibleElements('.keytable tr', 2);
    // Second row has check, first row input
    confirm(false, 1);
    confirm(true, 2);
    // Make current
    I.click('input.makecurrent', locate('.keytable tr').at(1));
    I.wait(1);
    // Confirm flip of input/check
    confirm(true, 1);
    confirm(false, 2);
    // Close and re-open to confirm persist
    I.click('Done');
    I.waitForInvisible('.keytable');
    I.waitForVisible('.btn[name="yourKeys"]');
    I.click('.btn[name="yourKeys"]');
    I.waitForVisible('.keytable .fa-check');
    I.seeNumberOfVisibleElements('.keytable tr', 2);
    // Still flipped
    confirm(true, 1);
    confirm(false, 2);
});