/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 * © 2017 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Greg Hill <greg.hill@open-xchange.com>
 */

/// <reference path="../../steps.d.ts" />

Feature('Settings - Key Management');

Before(async function (users) {
    await users.create();
    await users.create();  // Second user for revoke test
});

After(async function (users) {
    await users.removeAll();
});

const assert = require('chai').assert;

Scenario('View details of your keys', async function (I, users) {

    const o = {
        user: users[0]
    };
    const userdata = o.user.userdata;

    await I.setupUser(o, true);  // need advanced settings

    I.login('app=io.ox/mail', o);
    await I.verifyUserSetup(userdata);

    // Next, log in to settings
    await I.goToSettings();

    // Check yourkeys list has at least two keys
    I.waitForVisible('.btn[name="yourKeys"]');
    I.click('.btn[name="yourKeys"]');
    I.waitForVisible('#userkeytable');
    I.waitForVisible('.userpgpdetail');
    I.seeNumberOfVisibleElements('tr', 2);

    // Check key details
    I.click('.userpgpdetail');
    I.waitForVisible('.details dl');
    let keyDetail = await I.grabTextFrom('#privKeyDetail .modal-body');
    assert.include(keyDetail, 'Fingerprint', 'Key Detail includes fingerprint');

    // Check able to check signatures
    I.click('.btn[data-action="signatures"]');
    I.waitForVisible('#pkeyDetail');
    I.see('Public Keys Detail', '#pkeyDetail .modal-title');
    I.seeNumberOfVisibleElements('tr', 2);
    I.see('Positive', '[title="Positive certification of a User ID"]');

    // Close all
    I.click('#pkeyDetail [data-action="cancel"]');
    I.click('#privKeyDetail [data-action="cancel"]');
    I.click('.btn[data-action="cancel"]');
    I.logout();

});

Scenario('PGP change password', async function (I, users) {

    const o = {
        user: users[0]
    };
    const userdata = o.user.userdata;

    // Log in to settings
    I.login('app=io.ox/mail', o);
    await I.verifyUserSetup(userdata);

    // Next, log in to settings
    await I.goToSettings();

    I.see('Change password');

    I.click('Change password');

    I.waitForVisible('#oldpass');

    I.fillField('#oldpass', userdata.password);

    I.fillField('#newpass1', userdata.password + '2');

    I.fillField('#newpass2', userdata.password);

    I.see('Passwords not equal');

    I.pressKey('2');

    I.click('Change');

    I.waitForVisible('.io-ox-alert');

    I.see('Password changed', '.io-ox-alert');


});

Scenario('Upload public keys and verify available', async function (I, users) {

    const o = {
        user: users[0]
    };
    const userdata = o.user.userdata;
    await I.setupUser(o, true);  // need advanced settings

    I.login('app=io.ox/mail', o);
    await I.verifyUserSetup(userdata);

    // Next, log in to settings
    await I.goToSettings();

    I.waitForVisible('.btn[name="recipients"]');
    I.click('.btn[name="recipients"]');

    I.waitForElement('.publicKeyDiv [name="publicKey"]');

    I.executeAsyncScript(function (done) {
        $('[name="publicKey"]').on('change', function () {
                var files = this.files;
                if (files.length > 0) {
                    require(['oxguard/pgp/uploadkeys'], function (uploader) {
                        uploader.uploadExternalKey(files)
                        .always(function () {

                        });
                    });
                }
        });
        done();
    });
    I.attachFile('.publicKeyDiv [name="publicKey"]', 'e2e/testFiles/email-example-com.asc');
    I.wait(1);

    // Confirm uploaded, and check key details
    I.click('.refreshkeys');
    I.wait(1);
    I.seeNumberOfVisibleElements('tr', 1);
    I.see('Test User ', 'td');
    I.click('.oxguard_pubkeylist a[ids="6be6dc8 b2f16762"]');
    I.waitForVisible('#pkeyDetail .modal-title');
    I.see('Public Keys Detail', '.modal-title');
    I.seeNumberOfVisibleElements('#keyDetail', 2);

    I.click('Signatures');
    I.waitForVisible('.signatureTable');
    I.seeNumberOfVisibleElements('tr', 2);
    I.see('Positive', '.signatureTable');
    I.click('.modal[data-point="oxguard/pgp/signatureview"] [data-action="cancel"]');
    I.wait(1);

    I.click('#pkeyDetail [data-action="cancel"]');
    I.click('#pkeyList [data-action="cancel"]');

    // Start a mail to see if key properly loaded
    I.openApp('Mail');
    I.waitForVisible('.io-ox-mail-window .window-body .classic-toolbar');
    I.startEncryptedEmail(userdata);
    I.waitForVisible('.fa.fa-2x.encrypted');

    // Add uploaded key email address and confirm key found and properly labeled
    I.fillField('.io-ox-mail-compose div[data-extension-id="to"] input.tt-input', 'email@example.com');
    I.fillField('[name="subject"]', 'test');
    I.waitForVisible('.fa-key');
    I.moveCursorTo('i.fa-key', 5, 5);
    I.waitForVisible('.tooltip');
    I.see('Uploaded', '.tooltip');

    // Cleanup
    I.click('.io-ox-mail-compose-window button[aria-label="Save and close"]');
    I.waitForElement('.modal');
    I.click('Delete draft');
    I.waitForInvisible('.modal-backdrop');
    I.logout();
});

Scenario('Private Key Download and Upload', async function (I, users) {

    const o = {
        user: users[0]
    };
    const userdata = o.user.userdata;

    await I.setupUser(o, true);  // need advanced settings

    I.login('app=io.ox/mail', o);
    await I.verifyUserSetup(userdata);

    // Next, log in to settings
    await I.goToSettings();

    // First, download the current private key
    I.waitForVisible('.btn[name="yourKeys"]');
    I.click('.btn[name="yourKeys"]');
    I.waitForVisible('.delPriv');
    // Get the keyId from table and download
    let keyId = await I.grabAttributeFrom('#userkeytable tr', 'data-id')
    let downloadedKey = await I.downloadPrivateKey(keyId, userdata);

    // OK, delete the key
    I.click('.delPriv');
    I.waitForVisible('.modal[data-point="oxguard/settings/deletePrivate"]');
    I.click('.btn[data-action="delete"]');
    I.waitForVisible('#deletepass');
    I.insertCryptPassword('#deletepass', userdata);
    I.click('.btn[data-action="delete"]');

    // Upload dialog should appear now
    I.waitForVisible('.modal[data-point="oxguard/settings/uploadPrivate"]');

    // We have to manualy perform the upload here
    await I.executeAsyncScript(function(downloadedKey, userdata, done) {
        var formData = new FormData();
        formData.append('key', downloadedKey);
        formData.append('keyPassword', userdata.password);
        formData.append('newPassword', userdata.password);
        var url = ox.apiRoot + '/oxguard/keys?action=upload&respondWithJSON=true&session=' + ox.session;
        $.ajax({
            url: url,
            type: 'POST',
            data: formData,
            processData: false, // tell jQuery not to process the data
            contentType: false, // tell jQuery not to set contentType
            success: function (data) {
                done(data);
            },
            fail: function(e) {
                done(e);
            }
        });
    }, downloadedKey, userdata);

    // Cancel dialog since uploaded already
    I.click('Cancel');

    // Confirm key is now there and same
    I.click('#refreshuserkeys');
    I.wait(1);
    I.waitForVisible('.delPriv');
    let checkKeyId = await I.grabAttributeFrom('#userkeytable tr', 'data-id')
    assert.equal(checkKeyId[0], keyId[0], 'Key Ids should match');

    // Cleanup
    I.click('.btn[data-action="cancel"]');
    I.logout();

});

Scenario('Revoke Private Key', async function (I, users) {

    const o = {
        user: users[1]
    };
    const userdata = o.user.userdata;

    await I.setupUser(o, true);  // need advanced settings

    I.login('app=io.ox/mail', o);
    await I.verifyUserSetup(userdata);

    // Next, log in to settings
    await I.goToSettings();

    // Revoke the key
    I.waitForVisible('.btn[name="yourKeys"]');
    I.click('.btn[name="yourKeys"]');
    I.waitForVisible('.delPriv');
    I.click('.delPriv');
    I.waitForVisible('.modal[data-point="oxguard/settings/deletePrivate"]');
    I.click('.btn[data-action="revoke"]');
    I.waitForVisible('#revokepass');
    I.insertCryptPassword('#revokepass', userdata);
    I.selectOption('#settings-revokereason', 'KEY_SUPERSEDED');
    I.click('.btn[data-action="revoke"]');

    // Confirm drawn as revoked
    I.wait(1);
    let property = await I.grabCssPropertyFrom('tr', 'color');
    assert.isTrue((property[0].indexOf('rgb(255') > -1) || (property[0].indexOf('rgba(255') > -1), 'Row color should now be red');

    // Check detail
    I.click('.userpgpdetail');
    I.waitForVisible('.info-line.stripes-red');
    I.see('Revoked', '.info-line');

    // Cleanup
    I.click('#privKeyDetail [data-action="cancel"]');
    I.click('.btn[data-action="cancel"]');
    I.logout();

});

// Confirm element is checked or not in keytable row
function confirm(checked, location) {
    const { I } = inject();
    within(locate('.keytable tr').at(location), () => {
        if (checked) {
            I.seeElement('.fa-check');
        } else {
            I.seeElement('input.makecurrent');
        }
    })
}

Scenario('Select current key', async function (I, users) {

    const o = {
        user: users[0]
    };
    const userdata = o.user.userdata;

    await I.setupUser(o, true);  // need advanced settings

    I.login('app=io.ox/mail', o);
    await I.verifyUserSetup(userdata);
    await I.verifyUserSetup(userdata, true);  // Additional key

    // Next, log in to settings
    await I.goToSettings();

    // Open keys
    I.waitForVisible('.btn[name="yourKeys"]');
    I.click('.btn[name="yourKeys"]');
    I.waitForVisible('.keytable .fa-check');
    // Confirm 5 rows (2 keys plus divider)
    I.seeNumberOfVisibleElements('.keytable tr', 5);
    // First row has check, fourth row input
    confirm(true, 1);
    confirm(false, 4);
    // Make current
    I.click('input.makecurrent', locate('.keytable tr').at(4));
    I.wait(1);
    // Confirm flip of input/check
    confirm(false, 1);
    confirm(true, 4);
    // Close and re-open to confirm persist
    I.click('Close');
    I.waitForInvisible('.keytable');
    I.waitForVisible('.btn[name="yourKeys"]');
    I.click('.btn[name="yourKeys"]');
    I.waitForVisible('.keytable .fa-check');
    I.seeNumberOfVisibleElements('.keytable tr', 5);
    // Still flipped
    confirm(false, 1);
    confirm(true, 4);
    

});
