/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 * © 2017 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Greg Hill <greg.hill@open-xchange.com>
 */

/// <reference path="../../steps.d.ts" />

Feature('Smime: Send Emails');

Before(async function (users) {
    await users.create().then(async function (user) {
        await user.hasCapability('smime');
        await user.doesntHaveCapability('guard');
    });
    
});

After(async function (users) {
    await users.removeAll();
});

async function sendMail(I, subject, data, userdata, sign, unencrypted) {
    // Open compose
    I.sendEmail(userdata, subject, data, undefined, { smime_only: true, sign: sign, unencrypted: unencrypted });

    I.waitForVisible('.io-ox-mail-window .leftside');
    I.click('#io-ox-refresh-icon');
    I.waitForVisible('.io-ox-mail-window .leftside ul li.unread');
    I.click('.io-ox-mail-window .leftside ul li.unread');
    I.waitForVisible('.io-ox-mail-window .mail-detail-pane .subject');
    I.see(subject);

    // Decrypt
    if (!unencrypted) {
        I.waitForElement('.btn.oxguard_passbutton');
        I.waitForElement('.og_password input.form-control.password_prompt');
        I.insertCryptPassword('.og_password input.form-control.password_prompt', userdata);
        I.click('.btn.oxguard_passbutton');

        I.verifyDecryptedMail(subject, data, sign);
    }

}

Scenario('Compose and receive Smime encrypted email', async function (I, users) {

    const o = {
        user: users[0]
    };
    const userdata = o.user.userdata;
    
    // Log in as User 0
    I.login('app=io.ox/mail', o);
    I.waitForVisible('.io-ox-mail-window .window-body');

    await I.setupSmimeKey(o.user);

    var random = Math.floor((Math.random() * 100000));
    var subject = "Test Subject " + random;
    var data = "Test data " + random;

    await sendMail(I, subject, data, userdata);

    // OK, done
    I.logout();

});

Scenario('Compose and receive Smime signed only email', async function (I, users) {

    const o = {
        user: users[0]
    };
    const userdata = o.user.userdata;
    
    // Log in as User 0
    I.login('app=io.ox/mail', o);
    I.waitForVisible('.io-ox-mail-window .window-body');

    await I.setupSmimeKey(o.user);

    var random = Math.floor((Math.random() * 100000));
    var subject = "Test Subject " + random;
    var data = "Test data " + random;

    await sendMail(I, subject, data, userdata, true, true);

    I.waitForVisible('.guard_signed');

    // OK, done
    I.logout();

});

Scenario('Compose and receive Smime encrypted and signed email', async function (I, users) {

    const o = {
        user: users[0]
    };
    const userdata = o.user.userdata;
    
    // Log in as User 0
    I.login('app=io.ox/mail', o);
    I.waitForVisible('.io-ox-mail-window .window-body');

    await I.setupSmimeKey(o.user);

    var random = Math.floor((Math.random() * 100000));
    var subject = "Test Subject " + random;
    var data = "Test data " + random;

    await sendMail(I, subject, data, userdata, true);

    I.waitForVisible('.guard_signed');
    // OK, done
    I.logout();

});

// Testing sending in mobile views
async function sendMobile(I, users, device, landscape) {
    const o1 = {
        user: users[0]
    };

    const userdata = o1.user.userdata;

    await I.openMobile(device, landscape);

    I.login('app=io.ox/mail', o1);
    I.waitForVisible('.inline-toolbar [data-action="io.ox/mail/actions/compose"]');

    await I.setupSmimeKey(o1.user);
    
    // Test data
    var random = Math.floor((Math.random() * 100000));
    var subject = "Test Subject " + random;
    var data = "Test data " + random;

    // Open compose
    await I.sendMobileEmail(userdata, subject, data, undefined, { smime: true });

    I.wait(2);

    I.waitForVisible('.list-item.unread');
    I.tap('.unread .list-item-content');

    // Decrypt
    I.waitForElement('.btn.oxguard_passbutton');
    I.waitForElement('.og_password input.form-control.password_prompt');
    I.insertCryptPassword('.og_password input.form-control.password_prompt', userdata);
    I.click('.btn.oxguard_passbutton');

    // Verify decrypted
    I.verifyDecryptedMail(subject, data);
}

// Test with Android in landscape
Scenario('Smime: Mobile: Android (Landscape) - Compose and receive encrypted email', async function (I, users) {
    if (await I.amDisabled('mobile')) return;
    if (await I.amDisabled('smime')) return;
    await sendMobile(I, users, 'Pixel 2', true);
});

// Test with large iphone size
Scenario('Smime: Mobile: Iphone X - Compose and receive encrypted email', async function (I, users) {
    if (await I.amDisabled('mobile')) return;
    if (await I.amDisabled('smime')) return;
    await sendMobile(I, users, 'iPhone X');
});
