/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Greg Hill <greg.hill@open-xchange.com>
 */

define('oxguard/settings/views/deleteRecoveryView', [
    'io.ox/backbone/views',
    'io.ox/core/extensions',
    'io.ox/backbone/mini-views',
    'io.ox/backbone/views/modal',
    'io.ox/core/settings/util',
    'io.ox/core/yell',
    'oxguard/core/og_http',
    'oxguard/oxguard_core',
    'gettext!oxguard'
], function (views, ext, mini, ModalView, util, yell, og_http, core, gt) {

    'use strict';

    var POINT = 'oxguard/settings/reset',
        INDEX = 0;

    function open(node, baton, type) {
        return openModalDialog(node, baton, type);
    }

    function openModalDialog(node, baton, type) {

        return new ModalView({
            async: true,
            focus: 'input[name="active"]',
            point: POINT,
            title: gt('Delete Recovery'),
            width: 640,
            model: new Backbone.Model({ type: type })
        })
        .inject({
        })
        .build(function () {
        })
        .addCancelButton()
        .addButton({ label: gt('Delete'), action: 'delete' })
        .on('delete', function () {
            performDelete(this.model.get('type')).done(function () {
                ext.point('oxguard/settings/detail').invoke('draw', node, baton);  // redraw settings
            }).done(this.close).fail(this.idle).fail(yell);
        })
        .open();
    }

    ext.point(POINT).extend(
        {
            index: INDEX += 100,
            id: 'switch',
            render: function () {
                this.$body.append(
                    $('<p>').append(gt('The password recovery is used to reset your password if you forget it.  If you delete the recovery, you will not be able to reset your password.')),
                    $('<p>').append(gt('Note: Changing your password will restore the password recovery.'))
                );
            }
        }
    );

    function performDelete(type) {
        var def = $.Deferred();
        core.getPassword(gt('Please verify your password before deleting the recovery.'))
        .done(function (e) {
            var data = {
                userid: ox.user_id,
                cid: ox.context_id,
                password: e.password
            };
            core.metrics('settings', 'delete-recovery');
            og_http.post(ox.apiRoot + '/oxguard/login?action=deleterecovery&type=' + type, '', data)
            .done(function (e) {
                if (e && e.error) {
                    if (e.error === 'none') {
                        yell('error', gt('Unable to delete recovery.  Possibly wrong password for one or more keys, or no keys found'));
                        def.reject();
                        return;
                    }
                }
                yell('success', gt('Recovery deleted'));
                ox.guard.set('recoveryAvail', false);
                def.resolve();
            })
            .fail(function (e) {
                if (e.responseText.trim() === 'Bad password') {
                    yell('error', gt('Bad Password'));
                } else {
                    yell('error', e.responseText);
                }
                def.reject();
            });
        })
        .fail(function () {
            def.reject();
        });
        return def;
    }

    return {
        open: open
    };

});
