/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

define('oxguard/settings/pane', [
    'io.ox/core/extensions',
    'io.ox/backbone/views/extensible',
    'io.ox/core/settings/util',
    'oxguard/util',
    'io.ox/backbone/mini-views',
    'io.ox/core/notifications',
    'gettext!oxguard',
    'oxguard/oxguard_core',
    'settings!oxguard',
    'less!oxguard/settings/style'
], function (ext, ExtensibleView, util, guardUtil, mini, notifications, gt, core, settings) {
    'use strict';

    ext.point('oxguard/settings/detail').extend({
        index: 100,
        id: 'oxguardsettings',
        draw: function (baton) {
            var node = this;
            if (guardUtil.isGuardLoaded()) {
                drawGuardSettings(node, baton);
            } else {
                guardUtil.addOnLoaded(function () {
                    drawGuardSettings(node, baton);
                });
            }
            /**  Optional notification on save
            settings.on('save', function () {
                notifications.yell('success', gt('Saved'));
            });
            */
        }
    });

    var INDEX = 0;

    function drawGuardSettings(node, baton) {
        node.empty();
        guardUtil.hasSetupDone()
        .done(function () {
            node.addClass('guard-settings');
            node.append(
                new ExtensibleView({ point: 'oxguard/settings/detail/view', model: settings })
                    .inject({
                        getPasswordOptions: function () {
                            return ([
                                { label: gt('Ask each time'), value: 0 },
                                { label: gt('10 minutes'), value: 10 },
                                { label: gt('20 minutes'), value: 20 },
                                { label: gt('30 minutes'), value: 30 },
                                { label: gt('1 hour'), value: 60 },
                                { label: gt('2 hours'), value: 120 },
                                { label: gt('Session'), value: 99999 }
                            ]);
                        }
                    })
                    .build(function () {
                        this.listenTo(settings, 'change', function () {
                            settings.saveAndYell().then(
                                function ok() {
                                    //
                                },
                                function fail() {
                                    notifications.yell('error', gt('Could not save settings'));
                                }
                            );
                        });
                    })
                    .render().$el
            );
        })
        .fail(function () {
            require(['io.ox/core/capabilities'], function (capabilities) {
                if (capabilities.has('guard') && (capabilities.has('guard-mail') || capabilities.has('guard-drive'))) {
                    node.append(setupGuard(node, baton));
                } else if (capabilities.has('smime')) {
                    node.append(setupSmime(node, baton));
                }
            });
            return;
        });
    }

    ext.point('oxguard/settings/detail/view').extend(
        {
            id: 'header',
            index: INDEX += 100,
            render: function () {
                this.$el.addClass('io-ox-guard-settings').append(
                    util.header(gt('%s security settings', ox.guard.getName()))
                );
            }
        },
        {
            id: 'defaults',
            index: INDEX += 100,
            render: function () {
                if (guardUtil.isGuest() || (!guardUtil.hasGuardMailCapability() && !guardUtil.hasSmime())) return;   // Guests don't set defaults for sending
                var smime;
                if (core.hasSmime() && guardUtil.hasGuardMailCapability()) {
                    smime = util.checkbox('defaultSmime', gt('Default to using S/MIME for signing and encryption'), settings).css('display', settings.get('smime') ? 'block' : 'none').addClass('smimeDefaultOption');
                    smime.on('change', function () {
                        if (settings.get('defaultSmime')) {
                            settings.set('defaultInline', false);
                        }
                    });
                }
                var pgpInline;
                if (core.hasGuardMail()) {
                    pgpInline = util.checkbox('defaultInline', gt('Default to using PGP inline for new mails'), settings).addClass('guardAdvanced').css('display', settings.get('advanced') ? 'block' : 'none');
                    pgpInline.on('change', function () {
                        if (settings.get('defaultInline')) {
                            settings.set('defaultSmime', false);
                        }
                    });
                }
                this.$el.append(
                    $('<div class="guardDefaults">').append(
                        util.fieldset(
                            gt('Defaults'),
                            // default to encrypt outgoing
                            util.checkbox('defaultEncrypted', gt('Default to send encrypted when composing email'), settings),
                            // default add signature
                            util.checkbox('defaultSign', gt('Default adding signature to outgoing mails'), settings),
                            smime,
                            // inline
                            pgpInline,
                            util.compactSelect('defRemember', gt('Remember password default'), settings, this.getPasswordOptions())
                        )
                    )
                );
            }
        },
        {
            id: 'passwordManagement',
            index: INDEX += 100,
            render: function (baton) {
                this.$el.append(
                    util.fieldset(
                        gt('Password management'),
                        baton.branch('buttons', null, $('<div class="form-group buttons" id="passwordDiv">'))
                    )
                );
            }
        },
        {
            id: 'smime',
            index: INDEX += 100,
            render: function () {
                if (core.hasSmime() && core.hasGuardMail() && !core.isGuest()) {
                    var smime = util.checkbox('smime', gt('Enable S/MIME'), settings);
                    smime.on('change', function () {
                        if (settings.get('smime')) {
                            $('.smimeDefaultOption').show();
                        } else $('.smimeDefaultOption').hide();
                        updateAdvanced();

                    });
                    this.$el.append(
                        util.fieldset(
                            gt('S/MIME'),
                            // default to encrypt outgoing
                            smime
                        )
                    );
                }
            }
        },
        {
            id: 'Advanced',
            index: INDEX += 100,
            render: function () {
                if (!core.hasGuard()) {  // If the user only has smime (no guard), go ahead and draw key buttons
                    this.$el.append(drawAdvanced());
                    return;
                }
                var advancedBox = util.checkbox('advanced', gt('Show advanced settings'), settings);
                advancedBox.on('change', function () {
                    updateAdvanced();
                });
                this.$el.append(
                    util.fieldset(
                        gt('Advanced'),
                        // default to encrypt outgoing
                        advancedBox
                    )
                );
                this.$el.append(
                    $('<div class="guardAdvanced" style="display:' + (settings.get('advanced') ? 'block' : 'none') + '">')
                        .append(drawAdvanced()).append(drawAutocrypt()));
                updateAdvanced();
            }
        }
    );
    //
    // Buttons
    //
    ext.point('oxguard/settings/detail/view/buttons').extend(
        {
            id: 'changePassword',
            index: 100,
            render: function (baton) {
                this.append(
                    $('<button type="button" class="btn btn-default" id="changePassword">')
                    .text(gt('Change password'))
                    .on('click', openDialog)
                );

                function openDialog() {
                    ox.load(['oxguard/settings/views/changePasswordView', 'oxguard/settings/views/selectTypeView']).done(function (view, selectType) {
                        var node = $('.guard-settings');
                        selectType.open({
                            title: gt('Select type of keys to change'),
                            pgpAction: function () {
                                view.open(node, baton, 'pgp');
                            },
                            smimeAction: function () {
                                view.open(node, baton, 'smime');
                            }
                        });
                    });
                }
            }
        },
        {
            id: 'resetPassword',
            index: 200,
            render: function () {

                // check enabled
                if (!ox.guard.get('recoveryAvail') || ox.guard.get('new')) {  // If no recovery avail, then don't display reset button
                    return;
                }
                this.append(
                    $('<button type="button" class="btn btn-default">')
                    .text(gt('Reset password'))
                    .on('click', openDialog)
                );

                function openDialog() {
                    ox.load(['oxguard/settings/views/resetPasswordView', 'oxguard/settings/views/selectTypeView']).done(function (view, selectType) {
                        selectType.open({
                            title: gt('Select type of keys to reset'),
                            pgpAction: function () {
                                view.open('pgp');
                            },
                            smimeAction: function () {
                                view.open('smime');
                            }
                        });
                    });
                }
            }
        },
        {
            id: 'changeEmail',
            index: 300,
            render: function () {

                // check enabled
                if (!ox.guard.get('recoveryAvail') || ox.guard.get('new')) {  // If no recovery avail, then don't display reset button
                    return;
                }
                this.append(
                    $('<button type="button" class="btn btn-default">')
                    .text(gt('Set email address for reset'))
                    .on('click', openDialog)
                );

                function openDialog() {
                    ox.load(['oxguard/settings/views/secondaryEmailView', 'oxguard/settings/views/selectTypeView']).done(function (view, selectType) {
                        selectType.open({
                            title: gt('Select type of keys to change email reset'),
                            pgpAction: function () {
                                view.open('pgp');
                            },
                            smimeAction: function () {
                                view.open('smime');
                            }
                        });
                    });
                }
            }
        },
        {
            id: 'deleteRecovery',
            index: 400,
            render: function (baton) {
                // check enabled
                if (!ox.guard.get('recoveryAvail') || ox.guard.get('new') || ox.guard.getSettings().noDeleteRecovery || guardUtil.isGuest()) {  // If no recovery avail, then don't display reset button
                    return;
                }
                var button = $('<button type="button" class="btn btn-default guardAdvanced" id="deleteRecovery">')
                    //#. Option to remove the possibility of recovering your password
                    .text(gt('Delete password recovery'))
                    .on('click', openDialog);

                if (!settings.get('advanced')) {   // Only visible if advanced checked
                    button.css('display', 'none');
                }

                this.append(button);
                function openDialog() {
                    ox.load(['oxguard/settings/views/deleteRecoveryView', 'oxguard/settings/views/selectTypeView']).done(function (view, selectType) {
                        var node = $('.guard-settings');
                        selectType.open({
                            title: gt('Select type of keys to delete recovery'),
                            pgpAction: function () {
                                view.open(node, baton, 'pgp');
                            },
                            smimeAction: function () {
                                view.open(node, baton, 'smime');
                            }
                        });
                    });
                }
            }
        }
    );

    function drawAdvanced() {
        var div = $('<div class="guardAdvanced" id="advancedDiv">');
        $('#deleteRecovery').show();
        var downloadPublic = $('<div>');

        var downloadPublicButton = $('<button type="button" class="btn btn-default" name="downloadPublic">')
        .text(gt('Download my public key'))
        .on('click', downloadPublicKey);
        downloadPublic.append(downloadPublicButton);
        var userKeys = $('<button type="button" class="btn btn-default" name="yourKeys">')
        .text(gt('Your keys'))
        .on('click', openUserKeysDialog);
        function openUserKeysDialog() {
            ox.load(['oxguard/settings/views/userKeysView', 'oxguard/settings/views/selectTypeView', 'oxguard/settings/smime/privateView']).done(function (view, selectType, smimeView) {
                selectType.open({
                    title: gt('Select type of keys to list'),
                    pgpAction: view.open,
                    smimeAction: smimeView.open
                });
            });
        }
        var recipients = '';
        var autoCrypt = '';
        if (guardUtil.hasGuardMailCapability()) {  // Guests don't manage Public keys for recipients
            recipients = $('<button type="button" class="btn btn-default" name="recipients">')
            .text(gt('Public keys of recipients'))
            .on('click', openDialog);
            if (core.autoCryptEnabled()) {
                autoCrypt = $('<button type="button" class="btn btn-default" name="autoCrypt">')
                .text(gt('Autocrypt Keys'))
                .on('click', openAutoCryptDialog);
            }
        }
        function openDialog() {
            ox.load(['oxguard/settings/views/publicKeysView']).done(function (dialog) {
                dialog.open({
                    title: gt('Public keys of recipients')
                });
            });
        }
        function openAutoCryptDialog() {
            ox.load(['oxguard/settings/views/autoCryptKeysView']).done(function (view) {
                view.open();
            });
        }
        div.append(drawWipeOptions);
        var pgp = core.hasGuardMail();
        if (core.hasPGPandSmime()) {
            div.append(util.fieldset(
                gt('Keys'), userKeys));
            div.append(util.fieldset(
                gt('PGP'), downloadPublic, recipients, autoCrypt
            ));
        } else {
            div.append(util.fieldset(
                gt('Keys'), userKeys, pgp ? downloadPublic : '', pgp ? recipients : '', pgp ? autoCrypt : ''
            ));
        }

        div.append(drawVersion());
        return div;
    }

    function drawWipeOptions() {
        function getOptions() {
            return [{ label: gt('Keep emails decrypted until browser closed'), value: false },
                { label: gt('Do not keep emails decrypted.'), value: true }];
        }

        return util.fieldset(
            gt('Mail Cache'),
            new mini.CustomRadioView({ list: getOptions(), name: 'wipe', model: settings }).render().$el
        );
    }

    function drawAutocrypt() {
        if (!core.autoCryptEnabled() || !core.hasGuardMail()) return;
        var div = $('<div class="autocryptSettings guardAdvanced">');
        var transfer = $('<button type="button" class="btn btn-default" name="transferKeys">')
        .text(gt('Autocrypt transfer keys'))
        .on('click', openTransferDialog);
        function openTransferDialog() {
            ox.load(['oxguard/pgp/autocrypt/autocryptTransferView']).done(function (view) {
                view.open();
            });
        }
        var autoAdd = util.checkbox('autoAddAutocrypt', gt('Import autocrypt keys without asking'), settings);
        div.append(util.fieldset(gt('Autocrypt'),
            util.checkbox('enableAutocrypt', gt('Search incoming emails for keys in header'), settings),
            autoAdd, transfer));
        settings.on('change:enableAutocrypt', function () {
            checkEnabled(autoAdd);
        });
        checkEnabled(autoAdd);
        return div;
    }

    function checkEnabled(block) {
        var enabled = settings.get('enableAutocrypt');
        block.find('[name="autoAddAutocrypt"]').attr('disabled', !enabled);
        block.find('label').toggleClass('disabled', !enabled);
    }

    function downloadPublicKey() {
        require(['oxguard/api/keys'], function (keysAPI) {
            // current key
            keysAPI.downloadAsFile({ keyType: 'public' });
        });
    }

    function drawVersion() {
        if (_.device('small')) return ('');
        var div = $('<div class="guardVersion">');
        div.append(ox.guard.get('version') + '<br>build: ' + ox.guard.get('server'));
        return (div);
    }

    function updateAdvanced() {
        if (!settings.get('advanced')) {
            $('.guardAdvanced').hide();
            return;
        }
        $('.guardAdvanced').show();
        $('#advancedDiv').html(drawAdvanced());

    }

    // Convert old Guard server settings to local.  Should be able to remove version after 7.10.  Remove call from oxguard/register
    function initSettings(data) {
        settings.set('defaultEncrypted', data.pgpdefault === true);
        settings.set('defaultSign', data.pgpsign === true);
        settings.set('defaultInline', data.inline === true);
        settings.save();
    }

    function setupGuard(node, baton) {
        var div = $('<div>').addClass('io-ox-guard-settings');
        var passheader = util.header(gt('%s security settings', ox.guard.getName()));
        var instructions = $('<div>');
        //#. product name
        instructions.append('<p>' + gt('In order to start with %s security, you must set up a password for your secured files.', ox.guard.getName()) + '</p>');
        var startButton = $('<button type="button" class="btn btn-default guard-start-button">' + gt('Start') + '</button>');
        startButton.click(function () {
            require(['oxguard/core/createKeys'], function (keys) {
                core.metrics('settings', 'create-keys');
                keys.createKeysWizard()
                .done(function () {
                    ext.point('oxguard/settings/detail').invoke('draw', node, baton);
                });
            });
        });
        return (div.append(passheader).append(instructions).append(startButton));
    }

    function setupSmime(node, baton) {
        var div = $('<div>').addClass('io-ox-guard-settings');
        var passheader = util.header(gt('%s security settings', ox.guard.getName()));
        var instructions = $('<div>');
        //#. product name
        instructions.append('<p>' + gt('In order to start with %s security, you must upload an S/MIME key.', ox.guard.getName()) + '</p>');
        var startButton = $('<button type="button" class="btn btn-default guard-start-button">' + gt('Start') + '</button>');
        startButton.click(function () {
            require(['oxguard/settings/smime/smimeUpload'], function (keys) {
                keys.open()
                .done(function () {
                    ext.point('oxguard/settings/detail').invoke('draw', node, baton);
                });
            });
        });
        return (div.append(passheader).append(instructions).append(startButton));
    }

    return {
        initSettings: initSettings
    };
});
