/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

define('oxguard/mail/register', [
    'io.ox/core/extensions',
    'io.ox/backbone/views/actions/util',
    'oxguard/core/og_http',
    'oxguard/mail/checker',
    'io.ox/core/api/account',
    'io.ox/core/capabilities',
    'oxguard/mail_lock/util',
    'oxguard/mail/mail_metrics',
    'io.ox/mail/common-extensions',
    'oxguard/auth',
    'settings!oxguard',
    'gettext!oxguard',
    'oxguard/mail/mailReplyActions',
    'less!oxguard/pgp/style'
], function (ext, actionsUtil, http, checker, account, capabilities, util, metrics, mailExtensions, auth_core, settings, gt) {

    'use strict';

    // Draw security icons in mail detail view
    mailExtensions.security = function (baton) {
        if (!baton.model.get('security')) return;
        var security = baton.model.get('security');
        var pgpclass = 'icon-lock fa fa-lock pgp_superscript';
        if (security.decrypted) {
            if (security.type === 'SMIME') {
                var smime = $('<span class="oxguard_icon_fa" title="' + gt('This Email was encrypted with S/MIME.') + '"/>');
                smime.append($('<i class="icon-lock fa fa-lock" aria-hidden="true"/>'));
                this.append(smime);
            } else {
                var pgp = $('<span class="oxguard_icon_fa" title="' + gt('This Email was encrypted with PGP.') + '"/>');
                pgp.append('P<span style="font-size:0.5em;">gp</span>');
                var lockicon = $('<i class="' + pgpclass + '" aria-hidden="true"/>');
                pgp.append(lockicon);
                if (_.device('small')) {
                    pgp.css('margin-right', '5px');
                }
                this.append(pgp);
            }

        }
        if (security.signatures) {
            var verified = true;
            var error;
            var missing = false;
            security.signatures.forEach(function (sig) {
                verified = verified && sig.verified;
                if (sig.missing) missing = true;
                if (sig.error) error = sig.error;
            });
            var signedicon;
            if (verified) {
                if (security.type === 'SMIME') {
                    signedicon = $('<i class="oxguard_icon_fa fa-pencil-square-o fa guard_signed" aria-hidden="true" title="' + gt('This Email was signed and verified.') + '"/>');
                } else {
                    signedicon = $('<i class="oxguard_icon_fa fa-pencil-square-o fa guard_signed" aria-hidden="true" title="' + gt('This Email was signed and verified.') + '"/>');
                }

            } else if (!missing) {
                signedicon = $('<i class="oxguard_icon_fa_error fa-pencil-square fa guard_signed" aria-hidden="true" title="' + gt('This Email was signed and failed verification.') + '"/>');
            }
            if (missing) {
                signedicon = $('<i class="oxguard_icon_fa fa-pencil-square-o fa guard_signed" style="color:lightgrey" aria-hidden="true" title="' + gt('This Email was signed but unable to verify.') + '"/>');
            }
            if (error) {
                signedicon = $('<i class="oxguard_icon_fa_error fa-pencil-square fa guard_signed" aria-hidden="true" title="' + gt('Error verifying signature: %s', error) + '"/>');
            }
            if (_.device('small')) {
                signedicon.css('margin-right', '10px');
            }
            this.append(signedicon);
        }
        checkIntegrity(baton.data);
    };

    function checkIntegrity(data) {
        if (data && data.headers) {
            if (data.headers['X-Guard-Failed-Integrity']) {
                require(['io.ox/core/notifications'], function (notify) {
                    notify.yell('warning', gt('This email is missing one or more security checks.  HTML markup was removed for your security.'));
                });
            }
        }
    }

    // If so configured, cleanup any opened Guard emails
    function cleanGuardCache(baton) {
        if (settings.get('wipe') !== 'true') {  // If not configured to do so, exit
            return;
        }
        if (ox.guard && ox.guard.getAuth()) {  // Don't wipe if still authenticated
            return;
        }
        require(['io.ox/mail/api', 'io.ox/mail/main'], function (mailApi, mailMain) {
            if (!mailApi.pool) return;
            if (baton.options && baton.options.name === 'io.ox/mail/compose') {  // Don't wipe when just opening compose
                return;
            }
            var collection = mailApi.pool.get('detail');
            if (collection && collection.models) {
                collection.models.forEach(function (model) {
                    var security = model.get('security');
                    if (security && security.decrypted) {
                        var found = false;
                        if (baton.model) {  // Still within mail application
                            if (model.get('cid') === baton.model.cid) {  // Don't wipe the active mail
                                found = true;
                            }
                            var mailApp = mailMain.getApp();
                            if (mailApp.isThreaded()) {
                                mailApp.threadView.collection.models.forEach(function (thrModel) {
                                    if (thrModel.get('cid') === model.cid) {
                                        found = true;
                                    }
                                });
                            }
                        }
                        if (!found && model.get('origMail')) {
                            var orig = model.get('origMail');
                            model.clear();
                            model.set(orig);
                            var view = model.get('view');
                            if (view.$el) {  // when changing apps, need to redraw
                                view.redraw();
                            }
                        }
                    }
                });
            }
        });
    }

    if (capabilities.has('webmail')) {  // Register mail cleanup only if has mail
        ox.on('app:start', cleanGuardCache);
        ox.on('app:resume', cleanGuardCache);
    }

    ext.point('io.ox/mail/detail/header').extend({
        id: 'guardNotices',
        index: 'last',
        draw: function (baton) {
            if (settings.get('suppressInfo')) {
                return;
            }
            var security = baton.model.get('security');
            if (security) {
                var warn;
                var div = $('<div class="oxguard_info">');
                if (security && security.signatures) {
                    var sig = security.signatures[0];
                    if (sig && sig.missing) {
                        div.append($('<span>').append(gt('Unable to verify signature.  Missing public key.')));
                        warn = true;
                    }
                    if (sig && !sig.verified && !sig.missing) {
                        div.append($('<span class="signError">').append(gt('This Email was signed and failed verification.')));
                        warn = true;
                    }
                }
                if (warn) this.append(div);
            }

        }
    });

    ext.point('io.ox/mail/detail/attachments').extend({
        id: 'ogCheck',
        index: 1,
        draw: function (baton) {
            if (baton.data.security_info && baton.data.security_info.encrypted) baton.stopPropagation();
        }
    });

    ext.point('io.ox/mail/detail/body').extend({
        id: 'ogPGPCheck',
        index: 1,
        draw: function (baton) {
            if (this[0] !== undefined) {
                var location;
                if (this[0].host !== undefined) {  // Chrome
                    location = $(this[0].host);
                } else { // Others
                    location = this[0];
                }
                checker.checkPGP(baton, location);
            }
            if (baton.data.security_info && baton.data.security_info.encrypted) {
                if (!baton.data.unsupported) {
                    baton.stopPropagation();  // as long as we suport, stop further drawing
                    if (baton.data.security_info.type === 'PGP') checkAutocryptHeaders(baton);  // Check headers for the encrypted emails
                    cleanGuardCache(baton); // Clean previously decrypted emails
                }
            }
        }
    });

    // Check for autocrypt header
    ext.point('io.ox/mail/detail/body').extend({
        requires: 'guard-mail',
        index: 1000,
        id: 'autocrypt',
        draw: function (baton) {
            checkAutocryptHeaders(baton);
        }
    });

    ext.point('io.ox/mail/mobile/detail/body').extend({
        index: 230,
        id: 'ogPGPCheckMobile',
        draw: function (baton) {
            var location = this[0];
            checker.checkPGP.call(this, baton, location);
            baton.view.listenTo(baton.view.model, 'change:attachments', function () {
                checker.checkPGP(baton, location);
                if (baton.view.$el === null) return;
                if (!_.device('ios')) {
                    window.setTimeout(function () {
                        baton.view.$el.find('#oxgrpass').focus();
                    }, 50);
                }

            });
            if (baton.data.security_info && baton.data.security_info.encrypted) {
                baton.stopPropagation();
            }
        }
    });

    ext.point('io.ox/mail/detail/body').extend({
        index: 'last',
        id: 'cleanup',
        draw: function (baton) {
            cleanGuardCache(baton);
        }
    });

    //  Wipe previews for encrypted emails
    function wipePreview(baton) {
        if (baton.data.text_preview) {
            if (baton.data.content_type && baton.data.content_type.indexOf('encrypted') > 0) {
                baton.data.text_preview = '';
            }
            if (baton.data.text_preview.indexOf('----BEGIN PGP') > -1) {
                baton.data.text_preview = '';
            }
        }
    }

    ext.point('io.ox/mail/listview/item').extend({
        id: 'guard-preview',
        index: 10,
        draw: wipePreview
    });

    // Add lock to header
    ext.point('io.ox/mail/detail/header').extend({
        index: 230,
        id: 'pgpMailLock',
        draw: function (baton) {
            if (baton.data.decrypted) {
                // Add lock symbol to header
                var lockdiv = $('<div style="float:left; padding-right:5px;" id="lockdiv" class="oxguard_lock" ></div>');
                var lockstack = '<span class="icon-stack" title="' + gt('This Email was sent using PGP encryption. It has been decrypted with your private key') + '"><i class="icon-lock icon-large icon-stack-base fa fa-lock -fa-large oxguard_lock" />' +
                    '<i class ="icon-star-half-empty" style="color:black; margin-top:5px;"/></span>';
                lockdiv.append($(lockstack));
                if (baton.data.PGPResults && baton.data.PGPresults.verified === true) {
                    lockdiv.append('<i class="icon-edit" title="' + gt('This Email was signed by the sender and verified.') + '"/>');
                }
                $(this).prepend(lockdiv);
            }
        }
    });

    ext.point('io.ox/mail/view-options').extend({
        id: 'guardThread',
        index: 1,
        draw: function (baton) {
            if (capabilities.has('guest')) {
                baton.app.settings.set('threadSupport', false);
            }
        }
    });

    // Methods for actions
    // Check if attachment filename ends with .asc
    function isASC(e) {
        try {
            if (e.context instanceof Array) {
                for (var i = 0; i < e.context.length; i++) {
                    if (e.context[i].filename.indexOf('.asc') > 0) {
                        // Do not return positive for signatures.  Handled differently
                        if (e.context[i].content_type.indexOf('sig') > 0) return false;
                        return true;
                    }
                }
            }
            var filename = e.context.filename;
            if (filename === undefined) return false;
            if (e.context.content_type.indexOf('sig') > 0) return false;
            return (filename.indexOf('.asc') > 0);
        } catch (d) {
            console.log(d);
            return false;
        }
    }

    function checkAutocryptHeaders(baton) {
        if (baton.model && baton.model.get('headers')) {
            var headers = baton.model.get('headers');
            if (headers.Autocrypt || headers['Autocrypt-Gossip']) {
                require(['oxguard/pgp/autocrypt/autoCrypt'], function (autocrypt) {
                    if (headers.Autocrypt) {
                        autocrypt.check(baton.model)
                        .done(function () {
                            if (!settings.get('advanced')) return;  // Only users with advanced settings would have verified key
                            if (baton.data.security_info && baton.data.security_info.signed) {  //If signed, pull again after import
                                checker.pullAgain(false, baton);
                            }
                        });
                    }
                    if (headers['Autocrypt-Gossip']) {
                        autocrypt.checkGossip(baton.model);
                    }

                });
            }
        }
    }

    // test a single file if it is a key
    function testKey(file) {
        var test = {
            collection: null,
            context: file
        };
        return (isKey(test));
    }

    function isKey(e) {
        if (isASC(e)) {
            if (e.collection === null || e.collection.has('one')) { // only return true for individual files
                var type;
                var name;
                if (e.context instanceof Array) {
                    type = e.context[0].content_type;
                    name = e.context[0].filename;
                } else {
                    type = e.context.content_type;
                    name = e.context.filename;
                }
                if (type !== undefined) {
                    if (type.indexOf('application/pgp-keys') > -1) return true;
                }
                if (name === 'public.asc' || name === 'publickey.asc') return true;
                return (/[0x]*[0-9A-F]{8}\.asc/i).test(name); // Regex for 8 hex.asc
            }
        }
        return false;
    }

    function testAutoCrypt(file) {
        if (file && file.content_type === 'application/autocrypt-setup') return true;
        return false;
    }

    ext.point('io.ox/mail/detail/attachments').extend({
        id: 'Import',
        draw: function (e) {
            if (capabilities.has('guest')) return;
            var keyfound = false;
            var autoCryptStart = false;
            e.attachments.forEach(function (a) {
                if (testKey(a)) {
                    keyfound = true;
                }
                if (testAutoCrypt(a)) {
                    autoCryptStart = true;
                }
            });
            if (keyfound) {
                var keydiv = $('<div class="importKey">');
                var notice = $('<span>' + gt('PGP Public Key Found.  Click to Import') + '</span>');
                notice.click(function () {
                    doImportKey(e.attachments);
                });
                this.append(keydiv.append(notice));
            }
            if (autoCryptStart) {
                var aCkeydiv = $('<div class="importKey">');
                var aCnotice = $('<span>' + gt('AutoCrypt startup found.  Click to import key') + '</span>');
                aCnotice.click(function () {
                    require(['oxguard/pgp/autocrypt/autoCrypt'], function (autoCrypt) {
                        autoCrypt.doStart(e.attachments);
                    });
                });
                this.append(aCkeydiv.append(aCnotice));
            }
        }
    });

    new actionsUtil.Action('io.ox/mail/actions/save-encrypted-attachment', {
        capabilities: 'infostore guard-drive',
        matches: function (baton) {
            return !util.isSmime(baton.first()) && util.isDecrypted(baton.first());
        },
        collection: 'some',
        action: function (baton) {
            require(['io.ox/mail/actions/attachmentSave'], function (action) {
                var options = {
                    optPrompt: gt('Please re-enter your %s password.', ox.guard.getName()),
                    minSingleUse: true
                };
                auth_core.authorize(baton.data, options)
                .done(function () {
                    var list =
                         _.isArray(baton.data) ? baton.data : [baton.data];
                    for (var i = 0; i < list.length; i++) {
                        list[i].reEncrypt = true;
                    }
                    action.multiple(list);
                });
            });
        }
    });

    ext.point('io.ox/mail/attachment/links').extend({
        id: 'saveEncrypted',
        index: 550,
        mobile: 'high',
        // #. %1$s is usually "Drive" (product name; might be
        // customized)
        title: gt('Save encrypted to %1$s', gt.pgettext('app',
            'Drive')),
        ref: 'io.ox/mail/actions/save-encrypted-attachment'
    });

    function doImportKey(list) {
        if (ox.guard.needsKey()) {
            require(['oxguard/core/createKeys'], function (keys) {
                metrics.track('create-keys-to-import-public');
                keys.createKeysWizard()
                .done(function () {
                    loadPublicKey(list);
                });
            });
        } else {
            loadPublicKey(list);
        }
    }

    /// Sent folder extensions

    ext.point('io.ox/mail/links/inline').extend({
        index: 101,
        prio: 'lo',
        id: 'pinlink',
        title: gt('Check assigned PIN'),
        ref: 'io.ox/mail/actions/pin',
        mobile: 'lo'
    });

    var unified_sent = '';

    new actionsUtil.Action('io.ox/mail/actions/pin', {
        id: 'statusaction',
        matches: function (baton) {
            var e = baton.first();
            if (!(_.contains(account.getFoldersByType('sent'), e.folder_id)) && (e.folder_id !== unified_sent)) return (false);
            try {
                if (e.headers === undefined) return (false);
                return (e.headers['X-OxGuard-PIN'] !== undefined);
            } catch (ex) {
                console.log(ex);
                return (false);
            }
        },
        action: function (baton) {
            pin(baton);
        }
    });

    function pin(baton) {
        require(['io.ox/core/tk/dialogs'], function (dialogs) {
            var dialog = new dialogs.ModalDialog({ width: 300, center: true, enter: 'ok' });
            dialog.header($('<h4>').text(gt('PIN')));
            var pin = baton.first().headers['X-OxGuard-PIN'];
            var pintext = $('<h2>').append(pin);
            dialog.getBody().append(pintext);
            dialog.addPrimaryButton('ok', gt('OK'), 'ok').show();
        });
    }

    // We need to update the extensions that the email has security json
    ext.point('io.ox/mail/detail/attachments').extend({
        index: 1,
        id: 'guardDecrypted',
        draw: function (baton) {
            dupSecurity(baton);
        }
    });

    ext.point('io.ox/mail/mobile/detail/attachments').extend({
        index: 1,
        id: 'guardDecrypted',
        draw: function (baton) {
            dupSecurity(baton);
        }
    });

    ext.point('io.ox/mail/externalImages').extend({
        index: 1,
        id: 'checkAuth',
        perform: function (baton) {
            var def = $.Deferred();
            if (baton.data.security && baton.data.security.decrypted) {
                var options = {
                    optPrompt: gt('Please re-enter your %s password.', ox.guard.getName()),
                    minSingleUse: true
                };
                auth_core.authorize(baton, options)
                .done(function () {
                    def.resolve();
                })
                .fail(def.reject);
            } else {
                def.resolve();
            }
            return def;
        }
    });

    function dupSecurity(baton) {
        if (baton.data.security && baton.data.attachments.length > 0) {
            for (var i = 0; i < baton.data.attachments.length; i++) {
                baton.data.attachments[i].security = baton.data.security;
            }
        }
    }

    // Function to do authentication validation before calling an extension.
    // Checks if valid.  If not, prompts for password, calls extension again once auth
    function doAuthCheck(original, baton, storeAuth, minSingleUse) {
        baton.stopPropagation(); // Stop this extension and check authorization
        var callback = function (authData) {
            if (baton.data) {
                if (_.isArray(baton.data)) {
                    baton.data.forEach(function (d) {
                        d.auth = authData;
                    });
                } else {
                    baton.data.auth = authData;
                }
            }
            baton.resumePropagation();
            if (storeAuth) {
                if (_.isArray(baton.data)) {
                    baton.data.forEach(function (d) {
                        d.security = _.extend({ authentication: authData }, d.security);
                    });
                } else {
                    baton.first().security = _.extend({ authentication: authData }, baton.first().security);
                }
                // Guard-199, direct select of attachment adds list array
                if (baton.list && _.isArray(baton.list)) {
                    baton.list.forEach(function (d) {
                        d.auth = authData;
                        d.security = _.extend({ authentication: authData }, d.security);
                    });
                }
            }
            original.action(baton);
        };
        var options = {
            optPrompt: gt('Please re-enter your %s password.', ox.guard.getName()),
            minSingleUse: minSingleUse !== false,
            callback: callback
        };
        var d = _.isArray(baton.data) ? baton.data[0] : baton.data;
        // Still encrypted type
        if (d.security_info && d.security_info.type) {
            options.type = d.security_info.type;
        }
        // decrypted type
        if (d.security && d.security.type) {
            options.type = d.security.type;
        }
        auth_core.authorize(baton, options)
        .fail(function () {
            baton.stopPropagation();
            window.setTimeout(function () {
                baton.resumePropagation();
            }, 500);
        });
    }

    registerPasswordCheck('io.ox/mail/actions/reply', true);
    registerPasswordCheck('io.ox/mail/actions/reply-all', true);
    registerPasswordCheck('io.ox/mail/actions/forward', true);
    registerPasswordCheck('io.ox/mail/actions/edit', true);
    registerPasswordCheck('io.ox/mail/actions/edit-copy', true);
    registerPasswordCheck('io.ox/mail/actions/print', true);
    registerPasswordCheck('io.ox/mail/attachment/actions/view', true, 'default', false);
    registerPasswordCheck('io.ox/mail/attachment/actions/open');
    registerPasswordCheck('io.ox/mail/attachment/actions/download');
    registerPasswordCheck('io.ox/mail/attachment/actions/save');
    registerPasswordCheck('io.ox/mail/attachment/actions/vcard');
    registerPasswordCheck('io.ox/mail/attachment/actions/ical');

    // Function to check actions for Guard emails, and if found, proper authentication
    function registerPasswordCheck(extName, storeAuth, id, minSingleUse) {
        ext.point(extName).replace((id ? id : 'default'), function (original) {
            return {
                action: function (baton) {
                    if (storeAuth) {
                        if (util.isEncryptedMail(baton.first())) {
                            baton.first().security = {
                                type: util.getCryptoType(baton.first()),
                                decrypted: true
                            };
                            doAuthCheck(original, baton, storeAuth, minSingleUse);
                            return;
                        }
                    }
                    if (!util.isDecrypted(baton.first())) {
                        original.action(baton);
                        return;
                    }
                    doAuthCheck(original, baton, storeAuth, minSingleUse);
                }
            };
        });
    }

    /**
     * Extension to check if allowed to forward decrypted/encrypted email
     * Must have Guard-mail capability
     */
    ext.point('io.ox/mail/actions/forward').extend({
        id: 'guardLimitedReply',
        index: 10,
        matches: function (baton) {
            if (baton && baton.first()) {
                if (util.isDecrypted(baton.first()) || util.isEncryptedMail(baton.first())) {
                    if (!capabilities.has('guard-mail') && !util.isSmime(baton.first())) {
                        baton.stopPropagation();
                    }
                }
            }
            return false;
        }
    });

    ////////// End extensions
    /// Functions

    // Save public key from attachment
    function loadPublicKey(list) {
        _(list).each(function (data) {
            if (testKey(data)) { // make sure key
                var params = '&emailid=' + data.parent.id +
                '&attach=' + data.id +
                '&userid=' + ox.user_id +
                '&cid=' + ox.context_id +
                '&folder=' + data.parent.folder_id;
                if (data.security && data.security.decrypted) { //If was encrypted
                    require(['oxguard/auth'], function (auth) {
                        auth.authorize(list).then(function (auth) {
                            params = params + '&auth=' + encodeURIComponent(auth) +
                            '&inline=' + (data.security.pgpInline ? 'true' : 'false') +
                            '&filename=' + encodeURIComponent(data.filename);
                            doLoadPublicKey(params);
                        });
                    });
                } else {
                    doLoadPublicKey(params);
                }

            }
        });
    }

    function doLoadPublicKey(params) {
        var link = ox.apiRoot + '/oxguard/pgpmail/?action=savepublicattach' + params;
        http.get(link, '')
        .done(function (data) {
            require(['io.ox/core/notifications', 'oxguard/pgp/keyDetails'], function (notify, keydetails) {
                var keys = data.data.externalPublicKeyRings;
                var added = gt('Added keys: \r\n');
                for (var i = 0; i < keys.length; i++) {
                    added = added + gt('Key ID: ') + keys[i].ids + '\r\n';
                    if (keys[i].publicRing && keys[i].publicRing.keys) {
                        var details = keydetails.keyDetail(keys[i].publicRing.keys);
                        added = added + gt('User IDs: ') + details.keyids;
                    }
                }
                notify.yell('success', added.replace(/&lt;/g, '<').replace(/&gt;/g, '>'));
            });
        })
        .fail(function (e) {
            console.log(e);
            require(['io.ox/core/notifications'], function (notify) {
                if (e.status === 503) {
                    notify.yell('error', gt('Service Unavailable'));
                } else { notify.yell('error', e.responseText); }
            });
        });
    }

    return {
        registerPasswordCheck: registerPasswordCheck
    };
});
