/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

define('oxguard/mail/oxguard_mail_compose_core', [
    'io.ox/core/extensions',
    'oxguard/core/og_http',
    'io.ox/core/notifications',
    'oxguard/util',
    'settings!oxguard',
    'io.ox/core/tk/dialogs',
    'gettext!oxguard',
    'less!oxguard/mail/style'
], function (ext, og_http, notify, util, settings, dialogs, gt) {

    'use strict';

    //#. %s product name
    var guesticon = '<i id="newguest" class="fa fa-male oxguard_token key_tooltip" style="color:#5DCD55;" data-toggle="tooltip"/>';
    //#. Have encryption key - Shown on icon hover when a key symbol is displayed for recipient.  We/Guard have an encryption key for the recipient.
    var keyicon = '<i class="fa fa-key oxguard_token key_tooltip" aria-hidden="true" data-toggle="tooltip"></i>';
    //#. %s product name
    var failicon = '<i id="newguest" class="fa fa-ban oxguard_token" style="color:red;"/>';
    var prompting = false;

    // 7.8 compose code

    function drawKeyIcons(info, location) {
        var input = $(location).parent().find('.token-input');
        var inputwidth = input.width() - 30;
        var key = $(keyicon);
        if (info && info.result) {
            switch (info.result) {
                case 'pgp' :
                    if (info.key.keySource && info.key.keySource.trusted) {
                        key.addClass('trusted_key');
                    } else {
                        key.addClass('untrusted_key');
                    }
                    createTooltip(info.key, key, location);
                    $(location).find('.close').before(key);
                    break;
                case 'new' :
                    key.addClass('trusted_key');
                    $(location).find('.close').before(key);
                    createTooltip('new', key, location);
                    break;
                case 'guest' :
                    var gKey = $(guesticon);
                    $(location).find('.close').before(gKey);
                    createTooltip('guest', gKey, location);
                    break;
                case 'smime' :
                    key.addClass('trusted_key');
                    $(location).find('.close').before(key);
                    createTooltip(info, key, location);
                    break;
                case 'fail' :
                    $(location).find('.close').before(failicon);

                    break;
                default:
                    console.error('Unknown key type ' + info.result);
            }
        }
        input.width(inputwidth);
    }

    function createTooltip(data, loc, t) {
        require(['oxguard/pgp/keyDetails'], function (pubKeys) {
            var div = $('<div>');
            if (data.result === 'smime') {
                div.append(gt('S/MIME public certificate'), '<br>');
                div.append(gt('Expires: %s', new Date(data.key.expires).toLocaleDateString()), '<br>');
                div.append(gt('Serial: %s', data.key.serial));
            } else if (data === 'new') {  // New recipient
                div.append(gt('Key will be created upon sending.'));
            } else if (data === 'guest') {
                div.append(gt('Will create new guest account for %s', ox.guard.getName()));
            } else if (data === 'new') {  // New recipient
                div.append(gt('Key will be created upon sending.'));
            } else if (data === 'guest') {
                div.append(gt('Will create new guest account for %s', window.oxguarddata.productName));
            } else {
                var detail = pubKeys.keyDetail([data], true);
                div = $(detail.div).find('#keyDetail');
                div.append(gt('Source: ') + getSourceTranslation(data.keySource.name) + '<br/>');
                div.append(data.keySource.trusted ? gt('Trusted') : gt('Untrusted Source'));
                div.addClass(data.keySource.trusted ? 'trusted_key' : 'untrusted_key');
            }

            $(loc).tooltip({
                title: div.html(),
                trigger: 'hover focus',
                placement: 'bottom',
                html: true,
                delay: 200,
                animation: true,
                viewport: { selector: '.io-ox-mail-compose', padding: 10 }
            });
            var id = 'div' + (data.id ? data.id : Math.random());
            $(loc).append($('<div id="' + id + '" class="sr-only srdetails">').html(div.html()));
            t.attr('aria-describedby', id + ' ' + t.attr('aria-describedby'));
        });
    }

    function getSourceTranslation(source) {
        switch (source) {
            case 'HKP_PUBLIC_SERVER':
                return gt('Public server');
            case 'HKP_SRV_DNSSEC_SERVER':
                return gt('Secure DNS referred');
            case 'HKP_TRUSTED_SERVER':
                return gt('Trusted server');
            case 'HKP_SRV_SERVER':
                return gt('DNS referred');
            case 'HKP_CACHE':
                return gt('DNS referred');
            case 'GUARD_KEY_PAIR':
                return gt('System key');
            case 'GUARD_USER_UPLOADED':
                return gt('Uploaded key');
            case 'GUARD_USER_SHARED':
                return gt('Shared key');
            case 'WKS_SERVER':
                return gt('WebKey Directory Service');
            case 'WKS_SRV_DNSSEC_SERVER':
                return gt('WebKey DNSEC Service');
            case 'AUTOCRYPT_KEY':
                return gt('Autocrypt Keys');
            case 'AUTOCRYPT_KEY_USER_VERIFIED':
                return gt('Autocrypt Keys');
            default:
                return source;
        }
    }

    // Return a has of string
    function hash(str) {
        var hashed = 0;
        if (str.length === 0) return hashed;
        for (var i = 0; i < str.length; i++) {
            hashed = (31 * hashed + str.charCodeAt(i)) << 0;
        }
        return hashed;
    }

    function checkrecipient(baton) {
        if (!util.isGuardConfigured()) {
            if (!prompting) {
                prompting = true;
                require(['oxguard/oxguard_core', 'oxguard/core/createKeys'], function (core, keys) {
                    core.metrics('mail', 'create-keys-from-compose');
                    keys.createKeysWizard()
                    .done(function () {
                        checkrecipient(baton);
                    })
                    .fail(function () {
                        baton.view.securityModel.set('encrypt', false);
                    })
                    .always(function () {
                        prompting = false;
                    });
                });
            }
            return;// Do not check recipients until the master key is created (in case mailing to self)
        }
        require(['oxguard/mail/keymanager'], function (keyman) {
            keyman.checkAll(baton);
        });
    }

    function unlock(baton) {
        if (_.device('smartphone')) {
            $('.btn-primary:visible').html(gt('Send'));
        } else if (baton.view.app.get('window')) {
            var headerview = baton.view.app.get('window').nodes.header;
            var footerview = baton.view.app.get('window').nodes.footer;
            headerview.find('.btn-primary:visible').html(gt('Send'));
            if (footerview) {
                footerview.find('.btn-primary:visible').html(gt('Send'));
            }
        }
        baton.view.$el.find('.oxguard_key').hide();
        baton.view.$el.find('.oxguard_token').hide();
        baton.view.$el.find('.oxguard_token .srdetails').remove();
    }

    function lock(baton) {
        if (_.device('smartphone')) {
            $('.btn-primary:visible').html((settings.get('advanced') ? gt('Send Encrypted') : gt('Send Secure')));
        } else {
            // var headerview = baton.view.app.get('window').nodes.header;
            // var footerview = baton.view.app.get('window').nodes.footer;
            // headerview.find('.btn-primary:visible').html((settings.get('advanced') ? gt('Send Encrypted') : gt('Send Secure')));
            // if (footerview) {
            //     footerview.find('.btn-primary:visible').html((settings.get('advanced') ? gt('Send Encrypted') : gt('Send Secure')));
            // }
        }
        checkrecipient(baton);
        baton.view.$el.find('.oxguard_key').show();
        baton.view.$el.find('.oxguard_token').show();
    }

    return {
        lock: lock,
        hash: hash,
        unlock: unlock,
        drawKeyIcons: drawKeyIcons
    };
});
