/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

define('oxguard/core/guardModel', [
    'settings!oxguard',
    'io.ox/mail/sanitizer',
    'io.ox/core/capabilities'], function (settings, sanitizer, capabilities) {

    function sanitize(data) {
        return sanitizer.simpleSanitize(data);
    }

    // Parse out available languages, comparing Guard templates and UI available
    function checkLanguages(guardLanguages) {
        var available = {};
        if (!guardLanguages) return available;
        for (var lang in guardLanguages) {
            if (ox.serverConfig.languages[lang]) {
                available[lang] = guardLanguages[lang];
            }
        }
        return available;
    }

    var GuardModel = Backbone.Model.extend({

        defaults: {
            authcode: {},
            settings: {},
            loaded: false
        },

        initialize: function () {
            this.on('change:loaded', this.loadEvents);
        },
        /** Initialize with the Guard login data */
        loadData: function (data) {
            var model = this;
            Object.keys(data).forEach(function (key) {
                switch (key) {
                    case 'lang':
                        model.set('lang', checkLanguages(data[key]));
                        break;
                    case 'auth':
                        break;
                    default:
                        model.set(key, data[key]);
                }
            });
            this.set('loaded', true);
        },

        loadEvents: function () {  // Load any functions pending Guard loaded
            var startupList = this.get('onload');
            if (startupList) {
                try {
                    startupList.forEach(function (load) {
                        load.call();
                    });
                    this.set('onload', null);
                } catch (e) {
                    console.error('Problem executing onload functions for Guard ' + e);
                }
            }
        },

        getName: _.memoize(function () {
            var name = sanitize(ox.serverConfig['guard.productName'] || settings.get('productName'));
            return (!name || name === 'undefined') ? 'Guard' : name;
        }),

        // ****** Authentication  */
        setAuth: function (auth, type) {
            if (!type) type = 'pgp';
            if (type.toLowerCase() === 'pgp') {  // default Guard PGP
                switch (auth) {
                    case 'Bad Password':
                        return;
                    case 'Password Needed':
                        this.set('needsPassword', true);
                        return;
                    case 'No Key':
                        this.set('needsKey', true);
                        return;
                    default:
                        this.updateAuth(auth, type);
                        this.resetFlags();
                }
            } else {
                this.updateAuth(auth, type);
            }


        },
        resetFlags: function () {
            this.set('needsPassword', false);
            this.set('needsKey', false);
        },

        clearAuth: function (type) {
            this.resetFlags();
            this.updateAuth(null, type);
        },

        hasAuth: function (type) {
            return this.getAuth(type);
        },

        getAuth: function (type) {
            var authCodes = this.get('authcode');
            return authCodes[type];
        },

        hasAnyAuth: function () {
            var authCodes = this.get('authcode');
            return Object.entries(authCodes).length > 0;
        },

        /** Guard key status  */
        needsPassword: function () {
            return this.get('needsPassword') === true;
        },

        needsKey: function () {
            return this.get('needsKey') === true;
        },

        notReady: function () {
            return this.needsPassword() || this.needsKey();
        },
        updateAuth: function (auth, type) {
            if (!type) {
                this.set('authcode', {});
                return;
            }
            var currentAuth = this.get('authcode');
            currentAuth[type] = auth;
            this.set('authcode', currentAuth);
        },
        /* *Guard status **/
        isLoaded: function () {
            return this.get('loaded');
        },

        hasGuardMail: function () {
            return this.getSettings().oxguard && capabilities.has('guard-mail');
        },

        hasSmime: function () {
            return capabilities.has('smime');
        },
        hasSmimeKeys: function () {
            return this.get('smimeKeys');
        },
        getSettings: function () {
            return this.get('settings') || {};
        }
    });

    return GuardModel;
});
