/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

define('oxguard/core/createKeys', [
    'io.ox/backbone/views/modal',
    'oxguard/core/passwordView',
    'oxguard/api/keys',
    'gettext!oxguard',
    'less!oxguard/core/style'
], function (ModalDialog, PasswordView, keysAPI, gt) {

    'use strict';

    /**
     * @returns deferred object resolves to one of:
     *  * "OK" - everything is fine
     *  * "cancel" - user cancelled the dialog
     *  * object - failed with error message stored in responseText attribute
     */
    // Prompt for creating master keys for this user if not yet created.
    function openDialog() {
        var def = $.Deferred();

        new ModalDialog({
            async: true,
            point: 'oxguard/core/createKeys',
            title: gt('Create %s Security Keys', ox.guard.getName()),
            width: 640,
            enter: 'ok',
            focus: '#newogpassword'
        })
        .extend({
            'view': function () {
                this.view = new View({
                    model: new Model()
                });
                this.$body.append(
                    this.view.render().$el
                );
            },
            'button-state': function () {
                var button = this.$('button[data-action=ok]');
                // initial value
                button.attr({
                    // disable automatic state management of modal dialog
                    'data-state': 'manual',
                    'disabled': true
                });
                this.view.listenTo(this.view.model, 'change', function (model) {
                    button.attr('disabled', !model.isValid());
                });
            },
            'deferred': function () {
                this.view.listenTo(this.view.model, 'send', function (error) {
                    this.close();
                    return error ?
                        def.reject({ responseText: error }) :
                        def.resolve('OK');
                }.bind(this));
            }
        })
        .addButton({ label: gt('OK'), action: 'ok' })
        .addCancelButton()
        .on('cancel', function () {
            def.reject('cancel');
        })
        .on('ok', function () {
            this.view.model.send();
        })
        .on('dispose', function () {
            this.view.remove();
            this.view = null;
        })
        .open();

        return def;
    }

    function validateEmail(email) {
        var re = /^(([^<>()[\]\\.,;:\s@"]+(\.[^<>()[\]\\.,;:\s@"]+)*)|(".+"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
        return re.test(email);
    }

    var View = Backbone.View.extend({

        className: 'guard-create-key form-horizontal',
        initialize: function () {
            this.listenTo(this.model, 'change', this.handleChange);
            this.listenTo(this.model, 'before:send', this.busy);
            this.listenTo(this.model, 'send', this.idle);
            this.listenTo(this.model, 'send:error', this.handleError);

            this.$wait = $('<div class="og_wait" id="keygen">').append(
                $('<i class="fa-key fa icon-key fa-spin icon-spin"/>'),
                $('<span>').text(gt('Generating key, Please wait'))
            ).hide();
        },

        events: {
            'keyup #newogpassword': 'onPasswordChange',
            'input #newogpassword': 'onPasswordChange',
            'keyup #newogpassword2': 'onPasswordValidationChange',
            'input #newogpassword2': 'onPasswordValidationChange',
            'change input[name=recoverymail]': 'onRecoverymailChange'
        },
        onPasswordChange: function (ev) {
            this.model.set('password', ev.target.value);
        },
        onPasswordValidationChange: function (ev) {
            this.model.set('passwordValidation', ev.target.value);
        },
        onRecoverymailChange: function (ev) {
            this.model.set('email', ev.target.value);
        },

        handleChange: function (model) {
            var $el = this.$el;
            //reset error fields
            $el.find('.has-error')
                .removeClass('has-error')
                .find('.error-msg').empty();

            if (!model.isValid()) {
                model.validationError.forEach(function (error) {
                    $el.find(error.field)
                        .addClass('has-error')
                        .find('.error-msg').text(error.message);
                });
            }
            model.checkStrength();
            var strength = model.get('passwordStrength');
            $el.find('.password')
                .toggleClass('has-success', strength === 'good')
                .toggleClass('has-warning', strength === 'weak');
        },
        handleError: function (error) {
            require(['io.ox/core/notifications'], function (notify) {
                notify.yell('error', error);
            });
        },
        busy: function () {
            this.$wait.show();
        },
        idle: function () {
            this.$wait.hide();
        },
        render: function () {
            this.$el.empty().append(this.$wait);
            return this
                .renderDescription(this.$el)
                .renderPasswordPrompt(this.$el)
                .renderRecoverMailPrompt(this.$el);
        },
        renderDescription: function (el) {
            var prompt = '';
            if (this.model.get('initialSetup')) {
                prompt = gt('Please choose the password you will use for %s. You will need to type this password whenever you want to encrypt or decrypt items. Remember it should be different from your login password, and will not change if you change your login password.', ox.guard.getName());
            }
            if (this.model.get('prompt')) {
                prompt = this.model.get('prompt');
            }
            el.append($('<div>').append(
                $('<p>').text(prompt),
                $('<p>').text(gt('Please enter a password to protect your new encrypted items.'))
            ));
            return this;
        },
        renderPasswordPrompt: function (el) {
            el.append(
                $('<div>').addClass('form-group password').append(
                    $('<label for="newogpassword">')
                        .addClass('col-sm-12 col-md-4')
                        .text(gt('Password')),
                    $('<div>').addClass('col-sm-12 col-md-8').append(
                        new PasswordView.view({ 'id': 'newogpassword', 'validate': true }).getProtected()
                    ),
                    $('<div>').addClass('col-sm-12 error-msg')
                ),
                $('<div>').addClass('form-group password-validation').append(
                    $('<label for="newogpassword2">')
                        .addClass('col-sm-12 col-md-4')
                        .text(gt('Confirm')),
                    $('<div>').addClass('col-sm-12 col-md-8').append(
                        new PasswordView.view({ 'id': 'newogpassword2' }).getProtected()
                    ),
                    $('<div>').addClass('col-sm-12 error-msg')
                )
            );
            window.setTimeout(function () {
                if (_.device('desktop')) {
                    //$('#newogpassword').focus();
                } else {
                    $('[type="password"]').removeAttr('readonly');  // Remove the read only but do not focus.  User needs to click for keyboard
                }
            }, 1000);
            return this;
        },
        renderRecoverMailPrompt: function (el) {
            if (ox.guard.getSettings().noRecovery === true) {
                el.append($('<hr style="padding:10px;"/>'),
                    $('<p class="oxguard_warning">').text(gt('Warning: This password for encryption cannot be restored or recovered in any way.  If forgotten, all encrypted data will be lost')));
                return this;  // If no recovery set, don't add prompt for second password
            }
            if (this.model.get('initialSetup')) {
                el.append(
                    $('<hr style="padding:10px;"/>'),
                    $('<p>').text(gt('Please enter a secondary email address in case you need to reset your password.')),
                    $('<div>').addClass('form-group email').append(
                        $('<label>').addClass('col-sm-12 col-md-4').text(gt('Email:')),
                        $('<div>').addClass('col-sm-12 col-md-8').append(
                            $('<input name="recoverymail">').addClass('form-control')
                        ),
                        $('<div>').addClass('col-sm-12 error-msg')
                    )
                );
            }
            return this;
        }

    });

    var Model = Backbone.Model.extend({
        initialize: function () {
            var self = this;
            if (ox.guard.get('username')) return;
            // FIXME: should be handled differently (potential race conditions)
            require(['io.ox/core/api/user'], function (userAPI) {
                userAPI.getName().then(function (name) {
                    ox.guard.set('username', name);
                    self.set('name', name, { silent: true });
                });
            });

        },
        defaults: {
            name: ox.guard.get('username') || '',
            password: '',
            email: '',
            // config
            initialSetup: false,
            sent: false
        },
        send: function () {
            var self = this;
            this.trigger('before:send');
            if (!this.isValid()) {
                this.trigger('send send:error', this.validationError);
                return;
            }
            return keysAPI.create(this.pick('name', 'password', 'email')).then(function (data) {
                // TODO: maybe move to keysAPI
                ox.guard.clearAuth();
                ox.guard.set('recoveryAvail', !ox.guard.getSettings().noRecovery);
                this.set('sent', true);
                self.trigger('send send:ok');
                return data;
            }.bind(this), function (data) {
                // TODO: indicator for error that key already exists
                // gt('Problems creating keys. Keys already exist for this email address in another account.')
                self.trigger('send send:error', gt('Problems creating keys, please try again later.'));
                return data;
            });
        },
        validate: function (attrs) {
            var errors = [];
            var minlen = ox.guard.getSettings().min_password_length;
            if (minlen !== undefined && attrs.password !== undefined && attrs.password.length < minlen) {
                errors.push({
                    field: '.password',
                    message: gt('Passwords must be at least %s characters in length', minlen)
                });
            }
            if (attrs.password !== attrs.passwordValidation) {
                errors.push({
                    field: '.password-validation',
                    message: gt('Passwords not equal')
                });
            }
            if ((attrs.email.length > 1) && (!validateEmail(attrs.email))) {
                errors.push({
                    field: '.email',
                    message: gt('Enter new secondary Email address')
                });
            }
            return errors.length === 0 ? undefined : errors;
        },
        //handle password strength warnings (those not catched by validate) - adopted from oxquard/core/passwords.passwordCheck
        checkStrength: function () {
            var min = ox.guard.getSettings().min_password_length;
            var len = ox.guard.getSettings().password_length;
            if (min === undefined) min = 6;
            if (len === undefined) len = 10;

            if (len <= min) len = min + 1;
            if (!this.get('password')) return;

            if (this.get('password').length < min) {
                this.set('passwordStrength', 'bad');
                return;
            }

            var regex = /(?=^.{8,}$)(?=.*\d)(?=.*[!@#$%^&*]+)(?![.\n])(?=.*[A-Z])(?=.*[a-z]).*$/;
            if (this.get('password').match(regex)) {
                this.set('passwordStrength', 'good');
                return;
            }

            this.set('passwordStrength', 'weak');
        }
    });


    function createKeysWizard() {
        var def = $.Deferred();
        require(['io.ox/core/tk/wizard', 'oxguard/tour/main'], function (Tour) {
            Tour.registry.get('default/oxguard/createKeys').get('run')().then(def.resolve, def.reject);
        });
        return def;
    }

    return {
        createKeys: openDialog,
        createKeysWizard: createKeysWizard,
        CreateKeysModel: Model,
        CreateKeysView: View
    };

});
