/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

define('oxguard/auth', [
    'oxguard/oxguard_core',
    'oxguard/util',
    'gettext!oxguard'
], function (core, util, gt) {

    'use strict';

    var auth_core = {};

    // Perform auth against Guard server
    // Return auth code if any
    auth_core.authorize = function authorize(baton, o) {
        var options = _.extend({}, o);
        var def = $.Deferred();
        checkAuth(options, def);
        return def;
    };

    // Authentication for PGP items
    function doAuthCheck(options, def) {
        if (options.forceRelogin) {  // If we must relogin, show password prompt
            promptPassword(options, def);
        } else { // Else check if we have authentication already in session
            core.checkAuth(options)
            .done(function (data) {
                var found = false;
                data.forEach(function (d) {
                    if (!options.type || d.type.toLowerCase() === options.type.toLowerCase()) {
                        def.resolve(d.auth);
                        found = true;
                    }
                });
                if (!found) promptPassword(options, def);
            })
            .fail(function () {
                promptPassword(options, def);
            });
        }
    }
    function checkAuth(options, def) {
        // If we have known auth code, then proceed without deferred action (needed for tabs)
        if (util.hasStoredPassword()) {
            doAuthCheck(options, def);
            return;
        }
        ensureSetup().then(function () {
            doAuthCheck(options, def);
        }, def.reject);
    }

    function getPrompt(o) {
        if (o.optPrompt) return o.optPrompt;
        if (o && o.type) {
            if (core.hasPGPandSmime()) {
                if (o.type.toLowerCase() === 'smime') {
                    return gt('Enter %s S/MIME security password', ox.guard.getName());
                }
                return gt('Enter %s PGP security password', ox.guard.getName());
            }
        }
    }

    // Display password prompt to authorize
    function promptPassword(options, def) {
        var prompt = getPrompt(options);
        core.getPassword(prompt, true, options)  //Get password
        .done(function (passdata) {
            if (passdata.duration > -1 || options.minSingleUse) {
                core.savePassword(passdata.password, passdata.duration, options.type)
                .done(function (data) {
                    def.resolve(data.auth);
                    if (_.isFunction(options.callback)) {
                        options.callback(data.auth);
                    }
                })
                .fail(function (data) {
                    if (_.isFunction(options.failAuthCallback)) {
                        options.failAuthCallback();
                    }
                    handleFail(data.auth);
                    def.reject(data.auth, data);
                });
            } else {
                core.auth(ox.user_id, passdata.password, options.type)
                .done(function (data) {
                    if (data.auth.length > 20 && !(/Error:/i.test(data.auth))) {
                        if (_.isFunction(options.callback)) {
                            options.callback(data.auth);
                        }
                        def.resolve(data.auth);
                    } else {
                        if (_.isFunction(options.failAuthCallback)) {
                            options.failAuthCallback();
                        }
                        handleFail(data.auth);
                        def.reject(data.auth, data);
                    }
                });
            }
        })
        .fail(function () {
            def.reject('cancel');
        });
    }

    function ensureLoaded() {
        if (ox.guard.isLoaded()) return $.when();
        var def = $.Deferred();
        var counter = 0;
        var timer = setInterval(function () {
            if (ox.guard.isLoaded()) {
                def.resolve();
                clearInterval(timer);
            }
            if (counter++ > 100) {
                def.reject();
                clearInterval(timer); // not loading
                console.error('Response timeout from Guard server');
            }
        }, 100);
        return def;
    }

    function ensureSetup() {

        var def = $.Deferred();
        ensureLoaded().then(function () {
            if (!util.isGuardConfigured()) {
                require(['oxguard/core/createKeys'], function (keys) {
                    core.metrics('drive', 'create-keys');
                    keys.createKeysWizard()
                    .then(def.resolve, def.reject);
                });
                return;
            }
            // If keys were set up by receiving email, possible user hasn't had prompt to create password.
            if (ox.guard.needsPassword()) {
                require(['oxguard/core/tempPassword'], function (tempPass) {
                    var go = function () {
                        def.resolve();
                    };
                    tempPass.createOxGuardPasswordPrompt(null, go);
                });
            } else {
                def.resolve();
            }
        }, def.reject);

        return def;
    }

    function handleFail(auth) {
        require(['io.ox/core/notifications'], function (notifications) {
            switch (auth.toLowerCase()) {
                case 'bad password':
                    notifications.yell('error', gt('Bad password'));
                    return;
                case 'lockout':
                    notifications.yell('error', gt('Account Locked out'));
                    return;
                default:
                    notifications.yell('error', gt('Unspecified error') + '\n' + auth);
                    return;
            }
        });
    }

    auth_core.handleFail = handleFail;

    auth_core.ensureSetup = ensureSetup;

    return auth_core;
});
