/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

define('oxguard/api/util', [
    'oxguard/crypto/encr',
    'oxguard/util',
    'gettext!oxguard'
], function (encr, util, gt) {

    'use strict';

    var badCount = 0;

    function handleJsonError(error, errorResponse, baton) {
        errorResponse.errorMessage = gt('Error');
        errorResponse.retry = false;
        if (error.code !== undefined) {
            switch (error.code) {
                case 'GRD-PGP-0005':
                    if (ox.guard.hasAuth()) {  // If bad password based on auth, then clear auth and redo
                        ox.guard.clearAuth();
                        if (baton) baton.view.redraw();
                        return;
                    }
                    errorResponse.errorMessage = gt('Unable to decrypt Email, incorrect password.');
                    require(['io.ox/core/notifications'], function (notifications) {
                        notifications.yell('error', gt('Bad password'));
                    });
                    badCount++;
                    if (badCount > 2 && ox.guard.get('recoveryAvail')) {
                        errorResponse.errorMessage += ('<br/>' + gt('To reset or recover your password, click %s', '<a id="guardsettings" href="#">' + gt('Settings') + '</a>'));
                    }
                    errorResponse.retry = true;
                    break;
                default:
                    if (error.error !== undefined) {
                        errorResponse.errorMessage = error.error;
                    }
            }
        // No error code
        } else if (error.error !== undefined) {
            errorResponse.errorMessage = ox.guard.getName() + ': ' + error.error;
        }
    }

    return {
        encryptPasswords: function (data) {
            data = data || {};
            if (!ox.guard.get('pubKey')) return data;

            _.each(['password', 'encr_password', 'extrapass'], function (key) {
                // missing prop or empty value
                if (!(key in data) || !data[key]) return;
                var encrypted = encr.cryptPass(data[key]);
                if (!encrypted) return;
                var target = 'e_' + key;
                data[target] = encrypted;
                delete data[key];
            });
            return data;
        },

        format: util.format,

        showError: function (errorJson) {
            var errorResp = {};
            try {
                handleJsonError(errorJson, errorResp);
            } catch (e) {
                errorResp.errorMessage = gt('Unspecified error');
            }
            require(['io.ox/core/notifications'], function (notifications) {
                notifications.yell('error', errorResp.errorMessage);
            });
        }
    };

});
