/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Frank Paczynski <frank.paczynski@open-xchange.com>
 */

define('oxguard/api/auth', [
    'io.ox/core/http',
    'io.ox/core/event',
    'oxguard/api/util'
], function (http, Events, util) {

    'use strict';

    // https://documentation.open-xchange.com/components/guard/2.10.3/#tag/auth

    var encryptPasswords = util.encryptPasswords;

    // minutesValid
    // -1: or single use
    //  0: indefinite

    function getType(type) {
        return type ? type : '';
    }

    var api = {

        // Performs a login to obtain an authentication token
        // optionally: saves the token in the user session for specified period of time (minutesValid))
        authAndSave: function (data) {
            return http.POST({
                module: 'guard-json',
                params: {
                    action: 'auth-token',
                    type: getType(data.type),
                    time: new Date().getTime()
                },
                contentType: 'application/json; charset=utf-8;',
                data: JSON.stringify(encryptPasswords(data))
            }).fail(util.showError);
        },

        // Checks with the middleware if a valid authentication token exists within the session
        check: function (type) {
            api.trigger('before:check');
            return http.GET({
                module: 'guard-json',
                params: { action: 'auth-token', type: getType(type) }
            }).then(function (retdata) {
                // TODO: most of this "what situation do we have here?" should be handled on the server side
                if (!retdata || !retdata[0]) return $.Deferred().reject('none');
                // error cases
                retdata.forEach(function (data) {
                    var invalid = data.minutesValid === undefined || !_.isNumber(data.minutesValid) || !data.auth,
                        singleUse = data.minutesValid === -1;
                    if (singleUse) ox.guard.clearAuth(type);
                    if (invalid || singleUse) return $.Deferred().reject('exp');
                });
                // 99999: session
                api.trigger('success:check', retdata);
                return retdata;
            }).fail(function (e) {
                api.trigger('fail:check', e);
                if (_.isObject(e)) util.showError(e);
            });
        },

        // reset token in the user's session
        resetToken: function (type) {
            // TODO: should passcode be nulled in general?
            ox.guard.clearAuth(type);
            // destroy auth token
            return api.setToken('', 0, type);
        },

        // Saves an existing authentication token (obtained from a login
        // response) in the user's session for specified period of time.
        setToken: function (auth, time, type) {
            return http.PUT({
                module: 'guard-json',
                params: { action: 'auth-token', type: type ? type : 'pgp' },
                data: {
                    auth: auth,
                    type: getType(type),
                    minutesValid: time || -1
                }
            }).done(function (data) {
                return (data && data[0] ? data.auth : {});
            }).fail(function (e) {
                util.showError(e);
            });
        }
    };

    Events.extend(api);

    return api;
});
