/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.servlets.guest;

import static com.openexchange.java.Autoboxing.B;
import java.io.InputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import com.google.gson.JsonObject;
import com.openexchange.guard.antiabuse.GuardAntiAbuseService;
import com.openexchange.guard.common.auth.PasswordChangedResult;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.common.session.GuardUserSession;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.handler.ResponseHandler;
import com.openexchange.guard.inputvalidation.EmailValidator;
import com.openexchange.guard.inputvalidation.UUIDValidator;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.PasswordChangeService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.guest.exceptions.GuestServletExceptionCodes;
import com.openexchange.guard.storage.Storage;

/**
 * {@link GuestFirstPasswordAction}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v7.8.3
 */
public class GuestFirstPasswordAction extends GuardServletAction {

    /**
     * Converts the given{@link PasswordChangedResult} to JSON
     *
     * @param result The result to convert
     * @return The converted JSON
     */
    private JsonObject toJson(PasswordChangedResult result) {
        JsonObject json = new JsonObject();
        json.addProperty("auth", result.getAuth());
        json.addProperty("recovery", B(result.hasRecovery()));
        return json;
    }

    /**
     * For a guest users first access, change temporary password saved in recovery
     */
    @Override
    protected void doActionInternal(HttpServletRequest request, HttpServletResponse response, GuardUserSession userSession) throws Exception {
        GuardConfigurationService guardConfig = Services.getService(GuardConfigurationService.class);
        if (guardConfig.getBooleanProperty(GuardProperty.newGuestsRequirePassword, userSession.getUserId(), userSession.getContextId())) {  // If first password required, then we don't use this service
            throw GuestServletExceptionCodes.SERVICE_UNAVAILABLE.create();
        }
        GuardAntiAbuseService antiabuse = Services.getService(GuardAntiAbuseService.class);
        String ip = ServletUtils.getClientIP(request);
        if (antiabuse.isBad(ip)) {
            throw GuestServletExceptionCodes.LOCKOUT.create();
        }
        JsonObject json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request, userSession.getUserId(), userSession.getContextId());
        String newpass = null;
        String question = null;
        String answer = null;
        String email = null;
        String itemId = null;
        try {
            newpass = json.get("newpass").getAsString();
            question = json.get("question").getAsString();
            answer = json.get("answer").getAsString();
            email = new EmailValidator().assertInput(json.get("email").getAsString(), "email");
            itemId = new UUIDValidator("pgp-").assertInput(json.get("itemId").getAsString(), "itemId");
        } catch (NullPointerException e) {
            throw GuardCoreExceptionCodes.PARAMETER_MISSING.create("JSON requires newpass, question, answer, email, itemid");
        }
        String pin = null;
        if (json.has("pin")) {
            pin = json.get("pin").getAsString();
        }
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        GuardKeys keys = keyService.getKeys(email);
        if (keys == null) {
            antiabuse.addBad(ip);
            throw GuestServletExceptionCodes.NO_KEY.create(email);
        }
        if (keys.getContextid() > 0) {
            antiabuse.addBad(ip);
            throw GuestServletExceptionCodes.NOT_GUEST.create();
        }
        if (keys.getQuestion() != null && keys.getQuestion().equals("PIN")) {  // If PIN provided, check that it is valid
            antiabuse.addBad(ip);
            if (pin == null || !keys.getAnswer().equals(pin.trim())) {
                throw GuestServletExceptionCodes.PIN_REQUIRED.create();
            }
        }

        // Here, let's verify the itemId exists
        Storage storage = Services.getService(Storage.class);
        try (InputStream in = storage.readObj(itemId);) {
            if (in == null) {
                antiabuse.addBad(ip);
                throw GuestServletExceptionCodes.BAD_EMAIL_ID.create();
            }
        }

        // OK, verified.  Let's change the password
        PasswordChangeService passService = Services.getService(PasswordChangeService.class);
        OxCookie cookie = new OxCookie(request.getCookies());
        PasswordChangedResult result = passService.changePassword(keys.getUserid(), keys.getContextid(), email, "", newpass, question, answer, pin, cookie.getOxReaderID());
        ServletUtils.sendJson(response, toJson(result));
    }
}