/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.pgpcore.services.impl;

import org.bouncycastle.openpgp.PGPPublicKey;
import com.openexchange.exception.OXException;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.pgpcore.services.GuardKeyRingRetrievalStrategy;

/**
 * {@link CompositeGuardKeyRingRetrievalStrategy} searches for a recipient's Guard key ring using various strategies.
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class CompositeGuardKeyRingRetrievalStrategy implements GuardKeyRingRetrievalStrategy {

    final GuardKeyRingRetrievalStrategy[] strategies;

    /**
     * Initializes a new {@link CompositeGuardKeyRingRetrievalStrategy}.
     *
     * @param strategies The strategies using for finding a recipient's Guard key ring
     */
    public CompositeGuardKeyRingRetrievalStrategy(GuardKeyRingRetrievalStrategy... strategies) {
        super();
        this.strategies = strategies;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.pgpcore.commons.RecipientKeyRetrievalStrategy#getPublicKeyForRecipient(java.lang.String)
     */
    @Override
    public GuardKeys getKeyRingForRecipient(String recipient) throws OXException {
        for (GuardKeyRingRetrievalStrategy strategy : strategies) {
            GuardKeys key = strategy.getKeyRingForRecipient(recipient);
            if (key != null) {
                PGPPublicKey pgpKey = key.getPGPPublicKey();
                if (pgpKey != null) {
                    return key;
                }
            }
        }
        return null;
    }
}
