/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.milter.internal;

import java.awt.image.DataBuffer;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import org.slf4j.Logger;

/**
 * {@link MilterPacket}
 *
 * Creates a milter command packet, which has the command, data, and length
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.8.0
 */
public class MilterPacket {

    private static Logger Log = org.slf4j.LoggerFactory.getLogger(MilterPacket.class);

    private int command;

    private int dataLength, dataRead, headerLength;

    private ByteBuffer header, data;

    private boolean headerDone, dataDone;


    public MilterPacket () {
        this.header = ByteBuffer.allocate(5);
        this.data = null;
        this.command = 0;
        this.dataLength = 0;
        this.dataRead = 0;
        this.headerLength = 0;
        this.headerDone = false;
        this.dataDone = false;
    }

    private static int uByteToInt(byte b)
    {
        return (((int) b) & 0x0FF);
    }

    /*
     * Get the header of the data packet.  Always 5 characters.  4 lenght, one command
     */
    private void getHeader (ByteBuffer dataBuffer) {
        int toRead = Math.min(5 - headerLength, dataBuffer.remaining());
        for (int i = 0; i < toRead; i++) {
            header.put(dataBuffer.get());
            headerLength++;
        }
        // If we have it all, parse the length and command
        if (headerLength == 5) {
            header.flip();
            headerDone = true;
            this.dataLength = processLength() - 1;  // Length - 1 for command
            this.command = processCommand();
            if (this.dataLength == 0) {
                this.dataDone = true;
                this.data = null;
            } else
                this.data = ByteBuffer.allocate(dataLength);  // Initiate data buffer
        }
    }

    /**
     * Get the data from the buffer.  Should be equal to the length specified in the header
     * This may come in multiple parts
     * @param dataBuffer
     */
    private void getData (ByteBuffer dataBuffer) {
        int readableLength = Math.min(dataLength - dataRead, dataBuffer.remaining());
        this.data.put((ByteBuffer) dataBuffer.asReadOnlyBuffer().limit(dataBuffer.position() + readableLength));
        dataBuffer.position(dataBuffer.position() + readableLength);
        dataRead += readableLength;
        if (dataRead == dataLength) {
            data.flip();
            try {
                byte[] dataArray = data.array();
                String stringData = new String(dataArray, "UTF-8").toString();
            } catch (UnsupportedEncodingException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
            dataDone = true;
        }

    }

    /**
     * Get the 4 byte length of data
     * @param dataBuffer
     * @return
     */
    private int processLength () {
        int length = 0;
        for (int i = 0; i < 4; i++) {
            length <<= 8;
            length += uByteToInt(header.get());
        }
        return length;
    }

    // Call after length
    private int processCommand () {
        int command = uByteToInt(header.get());
        this.command = command;
        return command;
    }

    public boolean process (ByteBuffer dataBuffer) {

        if (!headerDone) {
            getHeader (dataBuffer);
        }

        if (headerDone && !dataDone) {
            if (dataBuffer.hasRemaining()) {
                getData (dataBuffer);
            }
        }

        if (dataDone && headerDone) return true;

        return false;

    }

    public int getCommand () {
        return this.command;
    }

    public ByteBuffer getData () {
        return this.data;
    }

    public void reset() {
        this.header = ByteBuffer.allocate(5);
        this.data = null;
        this.command = 0;
        this.dataLength = 0;
        this.dataRead = 0;
        this.headerLength = 0;
        this.headerDone = false;
        this.dataDone = false;
    }
}
