/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.milter.actions.encryptor;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import org.bouncycastle.openpgp.PGPPublicKey;
import com.openexchange.guard.keymanagement.commons.RecipKey;
import com.openexchange.guard.milter.service.MilterMail;
import com.openexchange.pgp.core.PGPEncrypter;
import com.openexchange.pgp.mail.tools.PGPMimeMailCreator;

/**
 * {@link DoEncrypt}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.8.0
 */
public class DoEncrypt {

    private String contentType = null;

    public byte[] encrypt (MilterMail mail) throws Exception {
        // Write the data as Mime Email
        MimeMessage message = getMessage(mail.getBodyBytes());
        ByteArrayOutputStream out = new ByteArrayOutputStream();

        // Add mime headers back
        addHeaders (mail, out);

        message.writeTo(out);
        InputStream in = new ByteArrayInputStream(out.toByteArray());
        ByteArrayOutputStream encrypted = new ByteArrayOutputStream();
        // List of recipients to array
        List<PGPPublicKey> pgpRecipientKeysList = new ArrayList<PGPPublicKey>();
        for (RecipKey recipientKey : mail.getKeys()) {
            pgpRecipientKeysList.add(recipientKey.getEncryptionKey());
        }
        PGPPublicKey[] pgpRecipientKeys = pgpRecipientKeysList.toArray(new PGPPublicKey[pgpRecipientKeysList.size()]);
        // Encrypt it
        new PGPEncrypter().encrypt(in, encrypted, true, pgpRecipientKeys);
        // Wrap it in new Mime Body
        MimeMessage wrapped = createWrapper(encrypted.toByteArray());
        contentType = wrapped.getContentType();
        ByteArrayOutputStream wrapped_out = new ByteArrayOutputStream();
        // Print it without headers to output
        wrapped.writeTo(wrapped_out, new String[] {"Content-Type", "MIME-Version", "Message-ID"});
        byte[] wrapped_bytes = wrapped_out.toByteArray();

        return wrapped_bytes;
    }

    private void checkAndAddHeader (String header, MilterMail mail, ByteArrayOutputStream out) throws IOException {
        if (mail.getHeader(header) != null) {
            String headerString = header + ": " + mail.getHeader(header) + "\r";
            out.write(headerString.getBytes(StandardCharsets.UTF_8));
        }

    }
    private void addHeaders (MilterMail mail, ByteArrayOutputStream out) throws IOException {
        checkAndAddHeader ("Message-ID", mail, out);
        checkAndAddHeader ("MIME-Version", mail, out);
        checkAndAddHeader ("Content-Type", mail, out);
        String space = "\r\r";
        out.write(space.getBytes(StandardCharsets.UTF_8));

    }

    private MimeMessage getMessage (byte[] data) throws MessagingException {
        MimeMessage message = new MimeMessage(Session.getDefaultInstance(new Properties()), new ByteArrayInputStream(data));
        return message;
    }

    private MimeMessage createWrapper (byte[] data) throws Exception {
        MimeMultipart mp = new PGPMimeMailCreator().createPGPMimePart(new ByteArrayInputStream(data), null);
        MimeMessage new_message = new MimeMessage(Session.getDefaultInstance(new Properties()));
        new_message.setContent(mp);
        new_message.saveChanges();
        return new_message;
    }

    public String getContentType() {
        return this.contentType;
    }
}
