/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 * © 2022 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Greg Hill <greg.hill@open-xchange.com>
 */

/// <reference path="../../steps.d.ts" />

Feature('Smime & PGP: Settings');

Before(async function (users) {
    await users.create().then(async function (user) {
        await user.hasCapability('smime');
    });
    
});

After(async function (users) {
    await users.removeAll();
});

// Select smime or pgp from select type popup
function select(smime) {
    const { I } = inject();
    I.waitForVisible('.modal[data-point="oxguard/settings/selectType"]');
    if (smime) {
        I.click('S/MIME');
    } else {
        I.click('PGP');
    }
    I.waitForInvisible('.modal[data-point="oxguard/settings/selectType"]');
}

function enableSmime() {
    const { I } = inject();
    I.waitForVisible('input[name="smime"]');
    I.click('Enable S/MIME');
}

Scenario('Can view PGP or Smime Private Keys, existing', async function (I, users) {

    const o = {
        user: users[0]
    };
    const userdata = o.user.userdata;

    await I.setupUser(o, true);  // need advanced settings

    I.login('app=io.ox/mail', o);
    // Create pgp and smime keys
    await I.verifyUserSetup(userdata);
    await I.setupSmimeKey(o.user);

    // Next, log in to settings
    await I.goToSettings();

    I.waitForVisible('.btn[name="yourKeys"]');
    I.click('.btn[name="yourKeys"]');

    // Should go right to PGP keys since Enable SMIME isn't checked
    I.waitForVisible('.keytable .fa-check');
    I.seeNumberOfVisibleElements('.keytable tr', 2);  // two keys displayed (main and sub)
    I.see('Details');   // PGP table has details

    I.click('Close');
    I.waitForInvisible('.keytable');

    enableSmime();

    // Now should be presented with option
    I.click('.btn[name="yourKeys"]');
    select(false);  // select PGP
    I.waitForVisible('.keytable .fa-check');
    I.seeNumberOfVisibleElements('.keytable tr', 2);  // two keys displayed (main and sub)
    I.see('Details');   // PGP table has details
    I.click('Close');

    I.click('.btn[name="yourKeys"]');
    select(true);

    I.waitForVisible('#userKeyList');
    I.seeNumberOfVisibleElements('.keytable tr', 1);
    I.see('Certifier');
    I.click('Done');
});

Scenario('Can view PGP or Smime Private Keys, smime missing', async function (I, users) {
    const o = {
        user: users[0]
    };
    const userdata = o.user.userdata;

    await I.setupUser(o, true);  // need advanced settings

    I.login('app=io.ox/mail', o);
    // Create pgp and smime keys
    await I.verifyUserSetup(userdata);

    // Next, log in to settings
    await I.goToSettings();

    enableSmime();

    I.click('.btn[name="yourKeys"]');
    select(false);  // select PGP
    I.waitForVisible('.keytable .fa-check');
    I.seeNumberOfVisibleElements('.keytable tr', 2);  // two keys displayed (main and sub)
    I.see('Details');   // PGP table has details
    I.click('Close');

    I.click('.btn[name="yourKeys"]');
    select(true);
    I.waitForVisible('#userKeyList');
    I.see('No keys found.');
    I.click('Done');
});

async function doResetTest (I, users, smime) {

    const o = {
        user: users[0]
    };
    const userdata = o.user.userdata;
    

    I.login('app=io.ox/mail', o);
    await I.setupSmimeKey(o.user);
    // Unique password to pgp
    userdata.password = 'pgpsecret';
    await I.verifyUserSetup(userdata);

    // Next, log in to settings
    await I.goToSettings();

    enableSmime();

    // Reset the password
    I.see('Reset password');
    I.click('Reset password');

    select(smime);

    I.waitForVisible('.modal-title');
    I.seeTextEquals('Reset Password', '.modal-title');  // Header
    I.click('.btn[data-action="reset"]');
    I.waitForInvisible('.modal-dialog');

    // Goto mail and wait for mail to arrive
    I.openApp('Mail');
    I.waitForVisible('.io-ox-mail-window .leftside');
    I.wait(1);
    I.click('#io-ox-refresh-icon');
    I.waitForVisible('.io-ox-mail-window .leftside ul li.unread');
    I.click('.io-ox-mail-window .leftside ul li.unread');

    //Pull the new temporary password from the email
    I.waitForVisible('.mail-detail-frame');
    I.switchTo('.mail-detail-frame');
    let newPass = await I.grabTextFrom('.bodyBox');
    I.switchTo();

    // Got back to settings and change the password
    await I.goToSettings();
    I.waitForVisible('#changePassword');
    I.click('#changePassword');

    select(smime);

    I.waitForVisible('#oldpass');
    I.fillField('#oldpass', newPass);
    I.insertCryptPassword('#newpass1', userdata);
    I.insertCryptPassword('#newpass2', userdata);
    I.click('.btn[data-action="change"]');

    // Confirm good change of the temporary password
    I.waitForVisible('.io-ox-alert-success');
    I.seeTextEquals('Password changed successfully.', '.io-ox-alert-success');

    I.logout();

};

Scenario('Reset PGP Password and Change when PGP and SMIME enabled', async function (I, users) {
    await doResetTest(I, users, false);
});

Scenario('Reset Smime Password and Change when PGP and SMIME enabled', async function (I, users) {
    await doResetTest(I, users, true);
});