/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

/* eslint-env node, es6  */
var defaultContext;

// please create .env file based on .evn-example
require('dotenv').config();

['LAUNCH_URL', 'PROVISIONING_URL', 'CONTEXT_ID'].forEach(function notdefined(key) {
    if (process.env[key]) return;
    console.error('\x1b[31m', `ERROR: Missing value for environment variable '${key}'. Please specify a '.env' file analog to '.env-example'.`);
    process.exit();
});

//  Parse the domain from launch URL to use in share configuration
function getShareDomain() {
    var url = process.env.LAUNCH_URL;
    if (url.indexOf('//') > 0) {
        url = url.substring(url.indexOf('//') + 2);
    }
    if (url.indexOf('/') > 0) {
        url = url.substring(0, url.indexOf('/'));
    }
    return url;
}


module.exports.config = {
    tests: './e2e/tests/**/*_test.js',
    timeout: 10000,
    output: './build/e2e/',
    helpers: {
        Puppeteer: {
            url: process.env.LAUNCH_URL,
            host: process.env.SELENIUM_HOST,
            smartWait: 1000,
            waitForTimeout: 10000,
            browser: 'chrome',
            chrome: {
                args: ['--no-sandbox']
            },
            restart: true,
            windowSize: '1024x800',
            uniqueScreenshotNames: true,
            timeouts: {
                script: 5000
            },
            // set HEADLESS=false in your terminal to show chrome window
            show: process.env.HEADLESS ? process.env.HEADLESS === 'false' : false
        },
        OpenXchange: {
            require: './e2e/helper',
            mxDomain: process.env.MX_DOMAIN,
            serverURL: process.env.PROVISIONING_URL,
            contextId: process.env.CONTEXT_ID,
            filestoreId: process.env.FILESTORE_ID || 6,
            smtpServer: process.env.SMTP_SERVER || 'localhost',
            imapServer: process.env.IMAP_SERVER || 'localhost',
            shareDomain: process.env.SHARE_DOMAIN || getShareDomain(),
            skipTests: process.env.SKIP_TESTS,
            newContext: parseInt(process.env.CONTEXT_ID, 10) + 1,
            admin: {
                login: process.env.PROVISIONING_USER || 'oxadminmaster',
                password: process.env.PROVISIONING_PASSWORD || 'secret'
            }
        },
        GuardHelper: {
            require: './e2e/helper.js'
        }
    },
    include: {
        I: './e2e/guardActor',
        users: './e2e/users',
        contexts: './e2e/context'
    },
    bootstrap: function (done) {
        // setup chai
        var chai = require('chai');
        global.assert = chai.assert;
        chai.config.includeStack = true;

        var codecept = require('codeceptjs'),
            config = codecept.config.get(),
            seleniumReady;
        seleniumReady = new Promise(function (resolve, reject) {
            if (config.helpers.WebDriver && /127\.0\.0\.1/.test(config.helpers.WebDriver.host)) {
                require('@open-xchange/codecept-helper').selenium
                    .start()
                    .then(resolve, reject);
            } else {
                resolve();
            }
        });

        const contexts = codecept.container.support('contexts'),
            helper = new (require('@open-xchange/codecept-helper').helper)();

        function getDefaultContext() {
            return helper.executeSoapRequest('OXContextService', 'list', {
                search_pattern: 'defaultcontext',
                auth: { login: 'oxadminmaster', password: 'secret' }
            }).then(function (result) {
                return result[0].return[0];
            });
        }
        function guessMXDomain(ctx) {
            return helper.executeSoapRequest('OXUserService', 'getData', {
                ctx,
                user: { name: 'oxadmin' },
                auth: { login: 'oxadmin', password: 'secret' }
            }).then(function (result) {
                return result[0].return.primaryEmail.replace(/.*@/, '');
            });
        }
        const testContextReady = getDefaultContext().then(function (ctx) {
            if (typeof config.helpers.OpenXchange.mxDomain !== 'undefined') return ctx.filestoreId;
            return guessMXDomain(ctx).then(mxDomain => {
                config.helpers.OpenXchange.mxDomain = mxDomain;
                return mxDomain;
            }).then(() => ctx.filestoreId);
        }).then(function (filestoreId) {
            if (typeof config.helpers.OpenXchange.filestoreId !== 'undefined') filestoreId = config.helpers.OpenXchange.filestoreId;
            const ctxData = {
                id: config.helpers.OpenXchange.contextId,
                filestoreId
            };

            function createDefaultContext() {
                return contexts.create(ctxData).then(function (ctx) {
                    defaultContext = ctx;
                }, function (err) {
                    console.error(`Could not create context ${JSON.stringify(ctxData, null, 4)}.\nError: ${err.faultstring}`);
                    if (Number(ctxData.id) === 10) {
                        console.error('Won\'t delete default context, use a different one.');
                        process.exit(1);
                    }
                    throw err;
                });
            }
            return createDefaultContext().catch(function () {
                console.warn('##--## Waiting 5s until context is removed. Press Ctrl+C to abort. ##--##');
                return new Promise(function (resolve) {
                    global.setTimeout(resolve, 5000);
                }).then(function () {
                    return helper.executeSoapRequest('OXContextService', 'delete', {
                        ctx: { id: ctxData.id },
                        auth: { login: 'oxadminmaster', password: 'secret' }
                    }).then(createDefaultContext);
                });
            });
        });

        Promise.all([
            seleniumReady,
            testContextReady
        ]).then(() => done())
        .catch(function (err) {
            console.error(err);
            done(err);
        });
    },
    teardown: function () {
        const helper = new (require('@open-xchange/codecept-helper').helper)();
        defaultContext.remove = function (auth) {
            helper.executeSoapRequest('OXContextService', 'delete', {
                ctx: { id: this.ctxdata.id },
                auth: auth || this.auth
            }).catch(e => console.log(e));
        };
        if (defaultContext.id !== 10) defaultContext.remove();
        //HACK: defer killing selenium, because it's still needed for a few ms
        setTimeout(function () {
            require('@open-xchange/codecept-helper').selenium.stop();
        }, 500);
    },
    plugins: {
        allure: { enabled: true },
        testrail: {
            require: './e2e/plugins/testrail',
            host: process.env.TESTRAIL_HOST || 'https://testrail.local',
            user: process.env.TESTRAIL_USERNAME || 'testuser',
            password: process.env.TESTRAIL_API_KEY || 'testkey',
            project_id: process.env.TESTRAIL_PROJECTID || '1',
            runName: process.env.TESTRAIL_RUNNAME || 'test',
            enabled: process.env.TESTRAIL_ENABLED || false
        }
    },
    name: 'Guard UI'
};

