/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 *  2016-2020 OX Software GmbH, Tarrytown, NY, USA. info@open-xchange.com
 *
 * @author Julian Baeume <julian.baeume@open-xchange.com>
 */

define('oxguard/util', [
    'io.ox/core/api/user',
    'settings!io.ox/mail',
    'io.ox/core/capabilities'
], function (userAPI, mailSettings, capabilities) {

    'use strict';

    var api = {};

    /**
     * Test if guard is configured.
     *
     * @return true if configured; false otherwise
     */
    api.isGuardConfigured = function () {
        if (!ox.guard.isLoaded()) return false;  // Make sure Guard response
        return !ox.guard.needsKey();
    };

    api.hasSetupDone = function () {
        var def = $.Deferred();
        if (ox.guard.needsKey()) {
            def.reject();
        } else if (!capabilities.has('guard') && capabilities.has('smime')) {
            if (ox.guard.hasSmimeKeys()) {
                def.resolve();
                return def;
            }
            require(['oxguard/api/certs'], function (keys) {
                keys.getCertificates().then(function () {
                    if (ox.guard.hasSmimeKeys()) {
                        def.resolve();
                    } else def.reject();
                });
            });
        } else { def.resolve(); }
        if (ox.guard.needsPassword()) {
            require(['oxguard/core/tempPassword'], function (core) {
                core.createOxGuardPasswordPrompt();
            });
        }
        return def;
    };

    api.hasCryptoCapability = function () {
        return (capabilities.has('guard') || capabilities.has('smime'));
    };

    api.hasGuardMailCapability = function () {
        return this.hasCryptoCapability() && (capabilities.has('guard-mail'));
    };

    api.isGuest = function () {
        return capabilities.has('guest');
    };

    api.hasSmime = function () {
        return capabilities.has('smime');
    };

    api.isGuardLoaded = function () {
        return ox.guard.isLoaded();
    };

    api.addOnLoaded = function (loadFunction) {
        if (!ox.guard.get('onload')) {
            ox.guard.set('onload', []);
        }
        ox.guard.get('onload').push(loadFunction);
    };

    api.hasStoredPassword = function () {
        return (ox.guard.hasAuth('PGP'));
    };

    api.format = function (fingerprint) {
        return (fingerprint || '').match(/.{1,4}/g).join(' ');
    };

    api.autoCryptEnabled = function () {
        return (ox.guard.getSettings().autoCryptEnabled);
    };

    api.getUsersPrimaryEmail = function () {
        var defaultSendAddress = $.trim(mailSettings.get('defaultSendAddress', ''));
        if (!defaultSendAddress) return $.when(defaultSendAddress);
        return userAPI.get().then(function (userData) {
            return userData.email1;
        });
    };

    api.sanitize = function (data) {
        var regex = new RegExp('<(?:!--(?:(?:-*[^->])*--+|-?)|script\\b(?:[^"\'>]|"[^"]*"|\'[^\']*\')*>[\\s\\S]*?</script\\s*|style\\b(?:[^"\'>]|"[^"]*"|\'[^\']*\')*>[\\s\\S]*?</style\\s*|/?[a-z](?:[^"\'>]|"[^"]*"|\'[^\']*\')*)>', 'gi');
        return (data.replace(regex, ''));
    };

    api.validateEmail = function (email) {
        var re = /^(([^<>()[\]\\.,;:\s@"]+(\.[^<>()[\]\\.,;:\s@"]+)*)|(".+"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
        return re.test(email);
    };

    return api;
});
