/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Greg Hill <greg.hill@open-xchange.com>
 */

define('oxguard/settings/views/secondaryEmailView', [
    'io.ox/core/extensions',
    'io.ox/backbone/views/modal',
    'io.ox/core/settings/util',
    'io.ox/core/yell',
    'oxguard/core/og_http',
    'oxguard/oxguard_core',
    'oxguard/core/passwordView',
    'oxguard/core/emailverify',
    'gettext!oxguard'
], function (ext, ModalView, util, yell, og_http, core, PasswordView, verify, gt) {

    'use strict';

    var POINT = 'oxguard/settings/secondary',
        INDEX = 0;

    function open(type) {
        return getSecondary(type).then(function (secondary) {
            openModalDialog(secondary, type);
        }, fail);
    }

    function fail(e) {
        yell('error', e.code === 'GUARD_FAILUER' ?
            gt('Unable to retrieve email address for password reset') :
            gt('Please retry later.')
        );
        throw e;
    }

    function openModalDialog(secondary, type) {

        return new ModalView({
            async: true,
            focus: 'input[name="active"]',
            point: POINT,
            title: gt('Secondary Email Address'),
            width: 640,
            model: new Backbone.Model({ type: type })
        })
        .inject({
            doChange: function () {
                var email = $('input[name="newemail"]');
                var verify = $('input[name="newemailverify"]');
                var password = $('#ogpass');
                return changeSecondary(email, verify, password, this.model.get('type'));
            },
            updateEmail: function () {
                this.$body.find('#secondaryEmail').text(secondary);
            }
        })
        .build(function () {

        })
        .addCancelButton()
        .addButton({ label: gt('Change email address'), action: 'change' })
        .on('change', function () {
            this.doChange().done(this.close).fail(this.idle).fail(yell);
        })
        .on('open', function () {
            this.updateEmail(this.model.get('type'));
        })
        .open();
    }

    ext.point(POINT).extend(
        {
            index: INDEX += 100,
            id: 'switch',
            render: function () {

                this.$body.append(
                    createSecondaryChange()
                );
            }
        }
    );

    function createSecondaryChange() {
        if (ox.guard.getSettings().noRecovery === true || ox.guard.get('recoveryAvail') === false) return;  // Don't display secondary email if no recovery is set
        var current = $('<span>').append(gt('Current email address:'));
        var currentEmail = $('<span id="secondaryEmail" style="padding-left:20px;">');
        var emailheader =
            $('<div class="oxguard_settings"/>')
            .append('<span> ' + gt('Change email address used for password reset.') + '</span><br/><br/>');
        if (_.device('small')) {
            emailheader.append(current).append('<br>').append(currentEmail);
        } else {
            emailheader.append(current).append(currentEmail);
        }
        var noSaveWorkAround = $('<input style="display:none" type="text" name="dontremember"/><input style="display:none" type="password" name="dontrememberpass"/>');
        var emailAddr = $('<input name="newemail" id="newemail" class="form-control"/>');
        var verifyPass = $('<input name="newemailverify" id="newemailverify" class="form-control">');
        var currentPrompt = $('<label for="ogpass">').append(gt('Enter current %s security password', ox.guard.getName()));
        var currentInput = new PasswordView.view({ 'id': 'ogpass', 'class': 'password_prompt' }).getProtected();
        var newPrompt = $('<label for="newemail">').append(gt('Enter new secondary email address'));
        var verifyPrompt = $('<label for="newemailverify">').append(gt('Verify email address'));
        var hint = $('<td>');
        var emailTable;
        if (_.device('small')) {
            emailTable = $('<div style="margin-top:15px;margin-left:10px;">')
            .append(currentPrompt).append('<br>').append(currentInput).append('<br>')
            .append(newPrompt).append('<br>').append(emailAddr).append('<br>')
            .append(verifyPrompt).append('<br>').append(verifyPass);
        } else {
            emailTable = $('<table/>')
            .append(noSaveWorkAround)
            //#. %s product name
            .append($('<tr>').append($('<td>').append(currentPrompt)).append($('<td>').append(currentInput)).append($('<td>')))
            .append($('<tr>').append($('<td>').append(newPrompt)).append($('<td>').append(emailAddr)).append(hint))
            .append($('<tr>').append($('<td>').append(verifyPrompt)).append($('<td>').append(verifyPass)).append($('<td>')));
        }
        var errorDiv = $('<div id="newemailerror" class="alert alert-info" style="display:none"/>');
        verify.setValidate(emailAddr, hint);
        verify.autoCompare(emailAddr, verifyPass, hint);
        return (emailheader.append(emailTable).append(errorDiv));
    }

    function getSecondary(type) {
        var def = $.Deferred();
        og_http.get(ox.apiRoot + '/oxguard/login?action=secondary', '&userid=' + ox.user_id + '&cid=' + ox.context_id + '&type=' + type)
        .done(function (d) {
            d = d.trim();
            if (d.length < 2) {
                require(['io.ox/core/api/account']).then(function (accountAPI) {
                    accountAPI.getPrimaryAddress().then(function (addr) {
                        if (addr.length < 2) {
                            def.reject();
                        } else {
                            def.resolve(addr[1]);
                        }
                    });
                });
            } else {
                def.resolve(d);
            }
        })
        .fail(function () {
            def.reject();
        });
        return def;
    }

    function changeSecondary(emailInput, verifyEmailInput, passwordInput, type) {
        var def = $.Deferred();
        var email = emailInput.val();
        var verifyEmail = verifyEmailInput.val();
        if (email !== verifyEmail) {
            showError(gt('Emails not equal'));
            def.reject();
            return def;
        }
        if (!verify.validate(email)) {
            showError(gt('Invalid email address'));
            def.reject();
            return def;
        }
        var data = {
            userid: ox.user_id,
            cid: ox.context_id,
            email: email,
            password: passwordInput.val()
        };
        core.metrics('settings', 'change-secondary-email');
        og_http.post(ox.apiRoot + '/oxguard/login?action=changesecondary', '&type=' + type, data)
            .done(function (data) {
                if (data.error) {
                    showError(data.error);
                    def.reject();
                    return;
                }
                showOK(gt('Successfully changed email address'));
                $('input[name="ogpass"]').val('');
                $('input[name="newemail"]').val('');
                $('input[name="newemailverify"]').val('');
                def.resolve();
            })
            .fail(function (d) {
                if (d.responseText.trim() === 'Bad password') {
                    showError(gt('Bad Password'));
                    if (!_.device('ios')) {
                        passwordInput.focus();
                    }
                }
                def.reject();
            });

        return def;
    }

    function showOK(msg) {
        require(['io.ox/core/notifications'], function (notify) {
            notify.yell('success', msg);
        });
    }

    function showError(msg) {
        require(['io.ox/core/notifications'], function (notify) {
            notify.yell('error', msg);
        });
    }

    return {
        open: open
    };

});
