/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

define('oxguard/settings/userKeys', [
    'oxguard/oxguard_core',
    'oxguard/api/keys',
    'settings!oxguard',
    'gettext!oxguard',
    'less!oxguard/pgp/style'
], function (core, keysAPI, settings, gt) {

    'use strict';

    // Generate list of public keys
    // - settings -> guard -> Your Keys

    var userKeyData = null;

    // register refresh listeners
    keysAPI.on('create', updateUserKeys);
    keysAPI.on('delete revoke', function (e, keyId) {
        // check if delete key is part of the table and refresh
        var node = $('#userkeytable' + ' [data-id="' + keyId + '"]');
        if (node.length) updateUserKeys();
    });

    // List of users private keys
    function userKeys() {
        var pubdiv = $('<div class="userKeyDiv"/>');
        var tableHeader = $('<b>').text(gt('Your key list'));
        var refresh = $('<a href="#" style="margin-left:10px;">')
        .attr('title', gt('Refresh'))
        .append($('<i class="fa fa-refresh fa-spin" id="refreshuserkeys" aria-hidden="true">'))
            .click(function (e) {
                $('#refreshuserkeys').addClass('fa-spin');
                updateUserKeyTable().done(function () {
                    $('#refreshuserkeys').removeClass('fa-spin');
                });
                e.preventDefault();
            });
        var addKey = $('<a href="#" style="margin-left:10px;">')
        .attr('title', gt('Add user key'))
        .append($('<i class="fa fa-plus" id="adduserkeys" aria-hidden="true">'))
        .click(function (e) {
            ox.load(['oxguard/settings/views/uploadPrivateView']).then(function (view) {
                view.open();
            });
            e.preventDefault();
        });
        tableHeader.append(refresh).append(addKey);
        var tablediv = $('<div class="keytable">');
        var table = $('<table id="userkeytable">');
        pubdiv.append(tableHeader.append(tablediv.append(table)));
        updateUserKeys();
        return (pubdiv);
    }

    function updateUserKeys() {
        var def = $.Deferred();
        $('#refreshuserkeys').addClass('fa-spin');
        updateUserKeyTable().done(function () {
            $('#refreshuserkeys').removeClass('fa-spin');
            def.resolve();
        });
        return def;
    }

    function updateUserKeyTable() {
        var deferred = $.Deferred();
        userKeyData = {};
        // Send the request to the oxguard server for the encrypted content key to decrypt
        keysAPI.getKeys().done(function (keyrings) {
            if (keyrings.length === 0) {
                // Keys deleted, no keys, open new key dialog
                ox.guard.setAuth('No Key');
                ox.load(['oxguard/settings/views/uploadPrivateView']).then(function (view) {
                    view.open();
                });
            }
            if (keyrings.length !== 0) {
                // create table and headers
                var table = $('<table id="userkeytable">').append(
                    [gt('Key ID'), gt('Private key'), gt('Current'), gt('Details'), gt('Delete'), gt('Download'), gt('Edit')].map(function (label) {
                        return $('<th>').text(label);
                    })
                );

                _.each(keyrings, function (keyring, index) {
                    _.each(keyring.publicRing.keys, function (data) {
                        // Save in cache
                        userKeyData[data.id] = data;

                        // add row for each key
                        table.append(
                            $('<tr>')
                                .attr({ 'data-id': data.id })
                                .addClass(data.revoked ? 'revoked' : '')
                                .addClass(data.expired ? 'expired' : '')
                                .append(
                                    // fingerprint
                                    $('<td data-name="fingerprint">').append(
                                        data.masterKey ? $() : $('<i class="fa fa-level-up fa-rotate-90" style="padding-right:7px; vertical-align:bottom;">'),
                                        $('<span>').text(data._short)
                                    ),

                                    // private key: check mark
                                    $('<td data-name="privatekey">').append(
                                        data.hasPrivateKey ? $('<i class="fa fa-check">').attr('title', gt('Has private key')) : $()
                                    ),

                                    // current: check mark OR checkbox
                                    $('<td data-name="current">').append(
                                        keyring.current ?
                                            $('<i class="fa fa-check">').attr('title', gt('Current')) :
                                            $('<input type="checkbox" class="makecurrent">').attr('title', gt('Make current'))
                                    ),

                                    // action: details
                                    $('<td data-name="details">').append(
                                        $('<a class="userpgpdetail" href="#">')
                                        .attr('title', gt('Details'))
                                        .append($('<i class="fa fa-search" aria-hidden="true">'))
                                    ),

                                    // action: delete
                                    $('<td data-name="delete">').append(
                                        !data.masterKey ? $() :
                                            $('<a href="#" class="delPriv">').attr({
                                                master: data.masterKey,
                                                private: !!data.hasPrivateKey,
                                                title: gt('Delete')
                                            })
                                            .append($('<i class="fa fa-trash" aria-hidden=true">'))
                                    ),

                                    // action: download
                                    $('<td data-name="download">').append(
                                        !data.masterKey ? $() :
                                            $('<a href="#" class="keydownload">').attr({
                                                title: gt('Download')
                                            })
                                            .append($('<i class="fa fa-download" aria-hidden="true">'))
                                    ),

                                    // action: edit
                                    $('<td data-name="edit">').append(
                                        !data.masterKey || !data.hasPrivateKey ? $() :
                                            $('<a href="#" class="keyedit">').attr({
                                                title: gt('Edit IDs')
                                            })
                                            .append($('<i class="fa fa-pencil" aria-hidden="true">'))
                                    )
                                )
                        );
                    });
                    // add spacer
                    if (index + 1 < keyrings.length) {
                        table.append(
                            $('<tr class="table-spacer">').append($('<td>').append($('<hr>')))
                        );
                    }
                });
                $('#userkeytable').replaceWith(table);

                table.on('click', '.userpgpdetail', function (e) {
                    var id = $(this).closest('tr').attr('data-id');
                    showLocalDetail(id);
                    e.preventDefault();
                });

                table.on('click', '.keydownload', function (e) {
                    var id = $(this).closest('tr').attr('data-id');
                    ox.load(['oxguard/settings/views/downloadPrivateView']).then(function (view) {
                        view.open(id);
                    });
                    e.preventDefault();
                });

                table.on('click', '.makecurrent', function (e) {
                    var id = $(this).closest('tr').attr('data-id');
                    makeCurrent(id);
                    e.preventDefault();
                });

                table.on('click', '.delPriv', function (e) {
                    var id = $(this).closest('tr').attr('data-id'),
                        master = $(this).attr('master'),
                        privateAvail = $(this).attr('private');
                    if (privateAvail === 'false') {
                        // If no private avail, we are just deleting a public key
                        ox.load(['oxguard/settings/views/deletePublicView']).then(function (view) {
                            view.open(id, privateAvail);
                        });
                    } else {
                        ox.load(['oxguard/settings/views/deletePrivateView']).then(function (view) {
                            view.open(id, master);
                        });
                    }
                    e.preventDefault();
                });

                table.on('click', '.keyedit', function (e) {
                    var id = $(this).closest('tr').attr('data-id');
                    ox.load(['oxguard/settings/views/editUserIdView']).then(function (view) {
                        view.open(id);
                    });
                    e.preventDefault();
                });
            } else {
                $('#userkeytable').replaceWith('<table id="userkeytable"><tr><td>' + gt('No Keys Found') + '</td></tr></table>');
            }

            deferred.resolve('OK');
            hideWait();
        })
        .fail(function () {
            hideWait();
        });
        return deferred;
    }

    function showLocalDetail(id) {
        var data = getCacheData(id);
        ox.load(['oxguard/settings/views/privateDetailView']).then(function (view) {
            view.open(data);
        });
    }

    function makeCurrent(id) {
        wait();
        // Send the request to the oxguard server for the encrypted content key to decrypt
        core.metrics('settings', 'change-current-key');
        keysAPI.setCurrentKey(id)
        .done(updateUserKeyTable)
        .fail(function (e) {
            require(['io.ox/core/notifications'], function (notify) {
                notify.yell('error', e.responseText);
                console.log(e);
                hideWait();
            });
        });

    }

    function wait() {
        $('#og_wait').show();
    }

    function hideWait() {
        $('#og_wait').hide();
    }

    function getCacheData(id) {
        var data = userKeyData[id];
        return (data);
    }

    return {
        userKeys: userKeys
    };

});
