/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

define('oxguard/settings/smime/privateView', [
    'io.ox/core/extensions',
    'io.ox/backbone/views/modal',
    'oxguard/oxguard_core',
    'oxguard/api/certs',
    'gettext!oxguard',
    'less!oxguard/pgp/style'
], function (ext, ModalView, core, certs, gt) {

    'use strict';

    var POINT = 'oxguard/settings/privateView',
        INDEX = 0;

    function open() {
        return openModalDialog();
    }

    function openModalDialog() {

        return new ModalView({
            async: true,
            point: POINT,
            title: gt('Your Certificates'),
            id: 'userKeyList',
            width: 640,
            model: new Backbone.Model()
        })
        .build(function () {

        })
        .addButton({ label: gt('Done'), action: 'done' })
        .on('done', function () {
            this.close();
        })
        .open();
    }

    ext.point(POINT).extend(
        // Draw Div with list keys
        {
            index: INDEX += 100,
            id: 'header',
            render: function () {
                var tableHeader = $('<b>' + gt('Your key list') + '</b>');
                var refresh = $('<a href="#" style="margin-left:10px;">')
                .attr('title', gt('Refresh'));
                var addKey = $('<a href="#" style="margin-left:10px;">')
                .attr('title', gt('Add user key'))
                .append($('<i class="fa fa-plus" id="adduserkeys" aria-hidden="true">'))
                .click(function (e) {
                    launchAddKeys();
                    e.preventDefault();
                });
                this.$body.append(tableHeader.append(refresh).append(addKey));
            }
        },
        {
            index: INDEX += 100,
            id: 'switch',
            render: function () {
                var div = $('<div class="keytable">');
                var node = this;
                updateTable().then(function (info) {
                    node.$body.append(
                        div.append(info)
                    );
                    addClickFunctions();
                });

            }
        }
    );

    function addClickFunctions() {
        $('.delPriv').on('click', function (e) {
            var id = $(this).attr('value');
            ox.load(['oxguard/settings/smime/deleteSmimeView']).then(function (view) {
                view.open(id).then(function () {
                    refreshKeys();
                });
            });
            e.preventDefault();
        });
        $('.downloadKey').on('click', function (e) {
            var id = $(this).attr('value');
            e.preventDefault();
            return core.getPassword(undefined, false).then(function (data) {
                core.verifyPassword(data.password, id, 'smime').then(function () {
                    certs.downloadAsFile({ password: data.password, serial: id, type: 'smime' }, data);
                });
            });

        });

        $('.makecurrent').on('click', function (e) {
            var id = $(this).attr('value');
            makeCurrent(id);
            e.preventDefault();
        });
    }

    function updateTable() {
        var newtable = $('<table id="smimekeytable">');
        var headers = $('<th>' + gt('Email') + '</th><th>' + gt('Expires') + '</th><th>' + gt('Current') + '</th><th>' + gt('Certifier') + '</th><th>' + gt('Download') + '</th><th>' + gt('Delete') + '</th>'); newtable.append(headers);
        var def = $.Deferred();
        certs.getCertificates().done(function (keyResponse) {
            if (keyResponse.certificates && keyResponse.certificates.length > 0) {
                var certs = keyResponse.certificates;
                certs.forEach(function (data) {
                    var tr = data.expired ? $('<tr style="color:red;">') : $('<tr>');
                    var td1 = $('<td style="text-align:left;">').append(data.email);
                    var td2 = $('<td>').append(new Date(data.expires).toDateString());
                    var td3 = $('<td>');
                    if (data.current) {
                        td3.append($('<i class="fa fa-check">').attr('title', gt('Current')));
                    } else {
                        var makecurrent = $('<input type="checkbox" class="makecurrent" value="' + data.serial + '">')
                            .attr('title', gt('Make current'));
                        td3.append(makecurrent);
                    }
                    var td4 = $('<td>').append(data.certifier);
                    var tr5 = $('<td>');
                    var download = $('<a href="#" class="downloadKey">')
                        .attr('value', data.serial)
                        .attr('title', gt('Download'))
                        .append($('<i class="fa fa-download" aria-hidden=true">'));
                    tr5.append(download);
                    var td6 = $('<td>');
                    var del = $('<a href="#" class="delPriv">')
                            .attr('value', data.serial)
                            .attr('title', gt('Delete'))
                        .append($('<i class="fa fa-trash" aria-hidden=true">'));
                    td6.append(del);
                    tr.append(td1, td2, td3, td4, tr5, td6);
                    newtable.append(tr);

                });
            } else if (keyResponse.certificates) {
                headers.remove();
                var link = $('<a class="noKeys">').append(gt('No keys found.  Please add keys by clicking here or using the add button above.'));
                link.on('click', function (e) {
                    e.preventDefault();
                    launchAddKeys();
                });
                newtable.append(link);
            }
            def.resolve(newtable);
        });

        return def;
    }

    function refreshKeys() {
        updateTable().done(function (updated) {
            $('#smimekeytable').replaceWith(updated);
            addClickFunctions();
        });
    }

    function launchAddKeys() {
        ox.load(['oxguard/settings/smime/smimeUpload']).then(function (view) {
            view.open().done(function () {
                refreshKeys();
            });
        });
    }

    function makeCurrent(id) {
        wait();
        certs.makeCurrent(id)
        .done(function () {
            refreshKeys();
        })
        .fail(function (e) {
            require(['io.ox/core/notifications'], function (notify) {
                notify.yell('error', e.responseText);
                console.log(e);
                hideWait();
            });
        });

    }

    function wait() {
        $('#og_wait').show();
    }

    function hideWait() {
        $('#og_wait').hide();
    }

    return {
        open: open
    };

});
