/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

define('oxguard/register', [
    'io.ox/core/extensions',
    'gettext!oxguard',
    'settings!oxguard',
    'io.ox/core/capabilities',
    'oxguard/core/guardModel',
    'less!oxguard/style'
], function (ext, gt, settings, capabilities, GuardModel) {

    'use strict';

    var oxguarddata = {
        passcode: null,
        settings:  {}
    };

    ox.guard = new GuardModel(oxguarddata);
    window.oxguarddata = oxguarddata;

    // Load data generated at compile (build version, etc)

    // Login section
    sendauth(''); // initial login, no password yet

    // Default settings if needed
    if (settings.get('advanced') === undefined) {
        settings.set('advanced', !!settings.get('defaultAdvanced'));
        settings.save();
    }

    // Sends first login to get settings info, key data
    function sendauth(pass) {
        require(['oxguard/oxguard_core'], function (oxguard) {
            var typeLogin = capabilities.has('guard') ? 'pgp' : 'smime';
            oxguard.auth(ox.user_id, pass, typeLogin)
            .done(function (data) {
                try {
                    require(['oxguard/oxguard_data'], function (generatedData) {
                        _.extend(data, generatedData);
                        ox.guard.loadData(data);
                    });

                    if (data.auth.length > 20 && !(/^Error:/i.test(data.auth))) { // If good code comes back, then store userid
                        ox.guard.setauth(data.auth, typeLogin);
                    } else { // If bad code, then log and mark oxguard.user_id as -1
                        if (/^((Error)|(Bad)|(Lockout))/i.test(data.auth)) {
                            ox.guard.clearAuth();
                        } else {
                            ox.guard.setAuth(data.auth, typeLogin);
                        }
                        if (data.error) {
                            require(['io.ox/core/notifications'], function (notify) {
                                notify.yell('error', ox.guard.getName() + '\r\n' + data.error);
                            });
                        }
                    }
                    if (settings.get('defaultEncrypted') === undefined) {
                        require(['oxguard/settings/pane'], function (defaults) {
                            defaults.initSettings(data.settings);
                        });
                    }
                    if (ox.context_id === 0) ox.context_id = data.cid; // If the cid wasn't loaded, update from backend

                    if (sessionStorage !== null) {
                        // If we were passed login info from login screen
                        try {
                            if (sessionStorage.oxguard && (sessionStorage.oxguard !== 'null')) {
                                var p = sessionStorage.oxguard;
                                sessionStorage.oxguard = null;
                                sendauth(p);
                            }
                        } catch (e) {
                            console.log('Private mode');
                            return;
                        }
                    }
                } catch (e) {
                    console.log(e);
                    console.debug('Unable to connect to the encryption server');
                }
            })
            .fail(function () {
                console.debug('Unable to connect to the encryption server');
            });
        });

    }

    function doLogout(notify) {
        var def = $.Deferred();
        ox.guard.clearAuth();
        require(['oxguard/api/auth'], function (authAPI) {
            // Destroy Guard mapping and auth-token
            authAPI.resetToken('none').done(function () {
                if (notify) {
                    require(['io.ox/core/notifications'], function (notify) {
                        notify.yell('success', gt('Logged out of %s', ox.guard.getName()));
                    });
                }
            });
        });

        return def;
    }

    ext.point('io.ox/core/appcontrol/right/help').extend({
        id: 'guard-guided-tour',
        index: 290,
        extend: function () {
            if (_.device('small') || capabilities.has('guest')) return;
            if (capabilities.has('guard-mail') || capabilities.has('guard-drive')) {
                this.append(
                    this.link('GuardTour', gt('Guided tour for %s', ox.guard.getName()), function (e) {
                        e.preventDefault();
                        require(['io.ox/core/tk/wizard', 'oxguard/tour/main'], function (Tour) {
                            Tour.registry.run('default/oxguard');
                        });
                    })
                );
            }
        }
    });

    ext.point('io.ox/core/appcontrol/right/account/signouts').extend({
        id: 'logoutOG',
        index: 50,
        extend: function () {
            this.link('logoutOG', gt('Sign out %s', ox.guard.getName()), function (e) {
                e.preventDefault();
                ox.guard.clearAuth();
                doLogout(true);
            });
            this.$el.on('shown.bs.dropdown', function () {
                if (ox.guard.hasAnyAuth()) {
                    $('[data-name="logoutOG"]').show();
                } else {
                    $('[data-name="logoutOG"]').hide();
                }
            });
        }
    });

    ext.point('io.ox/core/stages').extend({
        id: 'first',
        index: 107,
        run: function () {
            if (_.url.hash('i') === 'guestReset') {
                _.url.hash({ i: null, m: null, guardReset: 'true', app: 'io.ox/settings/main', folder: 'virtual/settings/oxguard' });

            }
        }
    });
});
