/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

define('oxguard/pgp/keyDetails', [
    'gettext!oxguard',
    'oxguard/core/og_http',
    'oxguard/oxguard_core',
    'oxguard/api/keys',
    'less!oxguard/style',
    'less!oxguard/pgp/style'
], function (gt, http, core, keysAPI) {

    'use strict';

    function hideWait() {
        $('#og_wait').hide();
    }

    var keyRingsCache = {};

    ////// Table generation

    // Generate list of public keys
    // - contact detail -> PGP Keys
    // - settings -> guard -> Public keys of recipients
    // - settings -> guard -> Autocrypt keys

    function listPublic(opt) {
        var o = _.extend({
            minimal: false,
            header: true,
            class: 'publicKeyDiv',
            title: gt('PGP Public Key List'),
            id: 'keyList'
        }, opt);

        var tableId = 'keylisttable' + o.id,
            container = $('<div class="key-container">').addClass(o.class).append(
                $('<div class="keytable">').append(
                // drawn/updated via 'updateKeyTable'
                    $('<table>').attr('id', tableId)
                )
            );

        if (o.header) {
            var tableHeader = $('<legend>').text(o.title);

            var refresh = $('<a href="#" style="margin-left:10px;" id="refreshkeys">').attr('title', gt('Refresh'))
                .append($('<i class="fa fa-refresh fa-spin refreshkeys" aria-hidden="true">'))
                .click(triggerRefresh);

            var fileinput = $('<input type="file" name="publicKey" style="display:none;" multiple>');
            var addKey = $('<a href="#" style="margin-left:10px;">')
                .attr('title', gt('Add Public Key'))
                .append($('<i class="fa fa-plus" id="addpgppublickeys" aria-hidden="true">'))
                .click(function (e) {
                    fileinput.unbind('change');
                    fileinput.on('change', function () {
                        var files = this.files;
                        if (files.length > 0) {
                            core.metrics('settings', 'upload-public-key');
                            require(['oxguard/pgp/uploadkeys'], function (uploader) {
                                uploader.uploadExternalKey(files)
                                .done(function () {
                                    updateKeyTable(o.id, o.folder_id, o.contact_id);
                                })
                                .always(function () {
                                    fileinput.val('');
                                });
                            });
                        }
                    });
                    fileinput.click();
                    e.preventDefault();
                });

            tableHeader.append(refresh);
            if (!o.minimal) tableHeader.append(fileinput).append(addKey);
            container.prepend(tableHeader);
        }

        update(o.id, o.folder_id, o.contact_id);

        // workaround: until we have a proper models and views
        keysAPI.on('create', triggerRefresh);
        keysAPI.on('delete revoke', function (e, keyId) {
            // check if delete key is part of the table and refresh
            var node = $('#' + tableId + ' [data-ids="' + keyId + '"]');
            if (node.length) triggerRefresh();
        });

        function triggerRefresh() {
            $('#refreshkeys > .fa').addClass('fa-spin');
            updateKeyTable(o.id, o.folder_id, o.contact_id).done(function () {
                $('#refreshkeys > .fa').removeClass('fa-spin');
            });
        }

        return container;
    }

    function getTable(keyRings, tableId, autoCrypt) {
        var showDelete = false;
        var newtable = $('<table class="keytable" id="keylisttable' + tableId + '">');
        //#. Table headers for list of keys, option for Details and to delete
        var headers = $('<th class="emailKey">' + gt('Email') + '</th><th>' + gt('Key Ids') + '</th><th class="keyDelete">' + gt('Delete') + '</th>'
                + (autoCrypt ? ('<th>' + gt('Verified') + '</th>') : ''));
        keyRingsCache = {};
        if (keyRings && keyRings.length > 0) {
            newtable.append(headers);
            for (var i in keyRings) {
                var tr = $('<tr>').attr({
                    'data-ids': keyRings[i].ids,
                    'data-autocrypt': autoCrypt
                });

                if (keyRings[i].guardKey === true || keyRings[i].owned === false) tr.addClass('oxguard_key_share');
                if (keyRings[i].expired === true) tr.css('color', 'red');
                var email = getUserIds(keyRings[i].publicRing);
                email = core.htmlSafe(email);
                var td1 = $('<td class="oxguard_pubkeylist emailKey" title="' + email + '">');
                td1.append(email);
                tr.append(td1);
                var td2 = $('<td class="oxguard_pubkeylist" title="' + email + ' ' + keyRings[i].ids + '">');
                td2.append($('<a href="#">')
                    .attr('ids', keyRings[i].ids)
                    .attr('notowned', ((keyRings[i].guardKey === true) || (keyRings[i].owned === false)) ? 'true' : 'false')
                    .on('click', detail)
                    .append(keyRings[i].ids));
                tr.append(td2);
                var td5 = $('<td class="keyDelete">');
                if ((keyRings[i].guardKey !== true) && (keyRings[i].owned === true)) {
                    var del = $('<a href="#">')
                    .append($('<i class="fa fa-trash" aria-hidden="true">'))
                    .attr({
                        'value': keyRings[i].ids,
                        'data-autocrypt': autoCrypt,
                        'tableid': tableId,
                        'title': gt('Delete')
                    })
                    .on('click', onDelete);
                    td5.append(del);
                    showDelete = true;
                }
                var td6 = '';
                if (autoCrypt) {
                    td6 = $('<td class="keyDownload">');
                    var verify = keyRings[i].verified ?
                        ($('<a href="#">')
                            .attr('ids', keyRings[i].ids)
                            .attr('verified', true)
                            .attr('title', gt('Verified'))
                            .append($('<i class="fa fa-check-square-o" aria-hidden="true">'))) :
                        ($('<a href="#">')
                            .attr('ids', keyRings[i].ids)
                            .attr('title', gt('Verify'))
                            .append($('<i class="fa fa-square-o" aria-hidden="true">')));
                    verify.click(verifyAutocrypt);
                    td6.append(verify);
                }
                tr.append(td5).append(td6);
                newtable.append(tr);
                keyRings[i].autoCrypt = autoCrypt;
                keyRingsCache[keyRings[i].ids] = keyRings[i];
            }
            if (!showDelete) {  // If only system keys, then hide the delete column
                newtable.find('.keyDelete').empty();
            }
            if (tableId) {
                $('.contactKeys' + tableId).show();
            }
            return newtable;
        }
        return ('<table id="keylisttable' + tableId + '"><tr><td>' + gt('No Keys Found') + '</td></tr></table>');
    }

    //  Shows details of keys in div
    //  Can be called for local keys, or for pgp public key list
    //  Minimal triggers display of details such as master, signing, etc or not
    function keyDetail(keys, minimal) {
        var keyids = '';
        var fingerprints = [];
        var keylong = ''; // This will be the key that has the ids (usually master)
        var images = null;
        var resultdiv = $('<div>');
        for (var i in keys) {
            var key = keys[i];
            //keyids = keyids + data[i].Key + ' ';  // List of key ids for changing share, inline (linked)
            if (i > 0) resultdiv.append('<hr>');
            var detaildiv = $('<div id="keyDetail"' + (minimal ? ' >' : ' style="padding-left:20px;">'));
            //#. if key is "expired"
            if (key.expired) {
                detaildiv.append('<b>' + gt('Expired') + '</b><br>');
                detaildiv.css('color', 'red');
            }
            if (key.revoked) detaildiv.append('<b style="color:red">' + gt('Revoked') + '</b><br>');
            detaildiv.append((key.privateKey ? ('<b>' + gt('Private Key Available') + '</b>') : gt('Public Key')) + '<br>');
            //#. Fingeprint of the key
            detaildiv.append(gt('Fingerprint:') + ' ' + formatFingerprint(key.fingerPrint) + '<br>');
            fingerprints.push(key.fingerPrint);
            detaildiv.append(gt('Created:') + ' ' + new Date(key.creationTime) + '<br>');
            if (!minimal) {
                //#. Key is a Master key
                detaildiv.append(gt('Master:') + ' ' + (key.masterKey ? gt('True') : gt('False')) + ',  ');
                //#. Key is an encryption key
                detaildiv.append(gt('Encryption:') + ' ' + (key.usedForEncryption ? gt('True') : gt('False')));
                if (key.usedForSigning) {
                    detaildiv.append(', ' + gt('Sign:') + ' ' + gt('True') + '<br>');
                } else {
                    detaildiv.append('<br>');
                }
                var ids = '';
                for (var j in key.userIds) {
                    if (j > 0) ids = ids + ', ';
                    ids = ids + core.htmlSafe(key.userIds[j]);
                }
                if (ids !== '') {
                    detaildiv.append(gt('IDs: ') + ids + '<br>');
                    keylong = key.id;  // The ID of the key with IDs for signature verification
                    keyids += ids;  // Accumulate IDs
                }
                if (key.images !== undefined) {
                    if (key.images.length > 0) {
                        images = key.images;
                        var image = $('<img alt="Embedded Image" id="imageID" src="' + key.images[0].imageData + '" style="float:right;max-height:50px;max-width:50px;"/>');
                        detaildiv.append(image);
                    }
                }
                if (key.validSeconds) {
                    detaildiv.append(gt('Expires: %1$s', new Date(key.creationTime + (key.validSeconds * 1000))) + '<br>');
                }
            }
            resultdiv.append(detaildiv);
        }
        return {
            div: resultdiv,
            keyids: keyids,
            keylong: keylong,
            images: images,
            fingerprints: fingerprints
        };
    }

    ////// Actions

    // Confirm deletion of item
    function onDelete(e) {
        e.preventDefault();
        var row = $(e.target).closest('tr'),
            data = {
                id: row.attr('data-ids'),
                keyType: row.attr('data-autocrypt') === 'true' ? 'autocrypt' : undefined
            };
        require(['oxguard/core/confirmView'], function (dialog) {
            dialog.open(gt('Please confirm you want to delete this key.'), data);
        });
    }

    // Verify autocrypt key action
    function verifyAutocrypt(e) {
        e.preventDefault();
        require(['oxguard/pgp/autocrypt/autoCrypt'], function (autocrypt) {
            autocrypt.verify(e);
        });
        core.metrics('settings', 'verify-autocrypt-key');
    }

    // Basic list of public key information
    // Shows keyDetail function output, plus adds buttons for sharing, etc in dialog
    // Shown with the inspect icon in pgp public key list
    function detail(e) {
        var id = $(this).attr('ids'),
            keyData = keyRingsCache[id];
        if (!keyData) return;
        var keydetail = keyDetail(keyData.publicRing.keys);
        ox.load(['oxguard/pgp/detailView']).then(function (dialog) {
            dialog.open(keydetail, keyData);
        });
        e.preventDefault();
    }

    function update(id, folder_id, contact_id) {
        $('.refreshkeys').addClass('fa-spin');
        updateKeyTable(id, folder_id, contact_id).done(function () {
            $('.refreshkeys').removeClass('fa-spin');
        });
    }

    // Switch depending on if specified contact
    function updateKeyTable(id, folder_id, contact_id) {
        var isContact = folder_id && contact_id,
            autoCrypt = id === 'autoCrypt',
            def = isContact ?
                keysAPI.getContactKeys(folder_id, contact_id) :
                keysAPI.getExternalPublicKeys(autoCrypt);

        return def.then(function (data) {
            $('#keylisttable' + id).replaceWith(getTable(data, id, autoCrypt));
            hideWait();
            return data;
        });
    }

    ///// Utilities

    function getUserIds(publicRing) {
        var ids = '';
        if (publicRing.keys && publicRing.keys.length > 0) {
            for (var i in publicRing.keys) {
                var key = publicRing.keys[i];
                if (key.userIds && key.userIds.length > 0) {
                    for (var k in key.userIds) {
                        if (key.userIds[k] && key.userIds[k].length > 1) {
                            ids += key.userIds[k] + ' ';
                        }
                    }
                }
            }
        }
        return ids;
    }

    function formatFingerprint(fp) {
        if (fp.length < 28) return (fp);
        if (fp.indexOf(' ') < 0) { // if not spaced
            var newFingerprint = '';
            for (var i = 0; i < fp.length; i = i + 4) {
                newFingerprint += fp.substring(i, i + 4) + ' ';
            }
            fp = newFingerprint;
        }
        var line1 = fp.substring(0, 24);
        var line2 = fp.substring(25, fp.length - 10) + '<b>' + fp.substring(fp.length - 10) + '</b>';
        return ('<span class="fingerPrint">' + line1 + '</span><br/><span class="fingerPrint">' + line2 + '</span>');
    }

    return {
        listPublic: listPublic,
        keyDetail: keyDetail
    };
});
