/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

define('oxguard/oxguard_core', [
    'settings!oxguard',
    'oxguard/api/auth',
    'oxguard/api/login',
    'oxguard/util',
    'io.ox/core/capabilities',
    'gettext!oxguard'
], function (guardSettings, authAPI, loginAPI, util, capabilities, gt) {

    'use strict';

    var badCount = 0;

    function metrics(app, action) {
        require(['io.ox/metrics/main'], function (metrics) {
            if (!metrics.isEnabled()) return;
            metrics.trackEvent({
                app: app,
                target: 'guard',
                type: 'click',
                action: action
            });
        });
    }

    // Escape html data to prevent cross scripting attack
    function htmlSafe(data) {
        var xss = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#39;',
            '/': '&#x2F;',
            '`': '&#x60;',
            '=': '&#x3D;'
        };
        return String(data).replace(/[&<>"'`=/]/g, function (s) {
            return xss[s];
        });
    }

    //// Failure Handling

    function handleJsonError(error, errorResponse, baton) {
        errorResponse.errorMessage = gt('Error');
        errorResponse.retry = false;
        if (error.code !== undefined) {
            switch (error.code) {
                case 'GRD-PGP-0005':
                    if (ox.guard.hasAuth()) {  // If bad password based on auth, then clear auth and redo
                        ox.guard.clearAuth();
                        if (baton) baton.view.redraw();
                        return;
                    }
                    errorResponse.errorMessage = gt('Unable to decrypt Email, incorrect password.');
                    require(['io.ox/core/notifications'], function (notifications) {
                        notifications.yell('error', gt('Bad password'));
                    });
                    badCount++;
                    if (badCount > 2 && ox.guard.get('recoveryAvail')) {
                        errorResponse.errorMessage += ('<br/>' + gt('To reset or recover your password, click %s', '<a id="guardsettings" href="#">' + gt('Settings') + '</a>'));
                    }
                    errorResponse.retry = true;
                    break;
                default:
                    if (error.error !== undefined) {
                        errorResponse.errorMessage = error.error;
                    }
            }

        }
    }

    function showError(errorJson) {
        var errorResp = {};
        try {
            handleJsonError(errorJson, errorResp);
        } catch (e) {
            errorResp.errorMessage = gt('Unspecified error');
        }
        require(['io.ox/core/notifications'], function (notifications) {
            notifications.yell('error', errorResp.errorMessage);
        });

    }

    function checkJsonOK(json) {
        if (json && json.error) {
            showError(json);
            return false;
        }
        return true;
    }

    /// Authentication Section

    // Password prompt
    function getPassword(message, timeOption, options) {
        var def = $.Deferred();
        require(['oxguard/core/passwordPrompt'], function (prompt) {
            prompt.open(message, timeOption, options)
            .done(function (data) {
                def.resolve(data);
            })
            .fail(function (e) {
                def.reject(e);
            });
        });
        return (def);
    }

    // Save a password in both auth code and token
    function savePassword(password, minutesValid, type) {
        // TODO: ensure calling methods use right value
        // 0 for session
        if (minutesValid > 9999) minutesValid = 0;
        return authAPI.authAndSave({
            password: password,
            minutesValid: minutesValid,
            type: type ? type : ''
        }).then(function (data) {
            if (data && data[0]) {
                storeAuth(data[0].auth, data[0].minutesValid, type);
                return data[0];
            }
        })
        .fail(function (e) {
            require(['io.ox/core/notifications'], function (notify) {
                switch (e.code) {
                    case 'GRD-MW-0003':
                        notify.yell('error', gt('Bad password'));
                        break;
                    case 'GRD-MW-0005':
                        notify.yell('error', gt('Account locked out due to bad attempts.  Please try again later.'));
                        break;
                    default:
                        notify.yell('error', gt('Error checking password') + '\r\n' + e.error);
                        break;
                }
                console.error(e);
            });
        });
    }

    // Store the auth locally for a time
    function storeAuth(auth, duration, type) {
        if (duration >= 0) {
            ox.guard.setAuth(auth, type);
            var time = duration * 1000 * 60;
            if (time > 0) {  // 0 is indefinite
                window.setTimeout(function () {
                    ox.guard.clearAuth(type);
                }, time);
            }
        }
    }

    // Check if authentication token in session and valid
    var checkAuth = (function () {

        // TODO: maybe it's possible to use api-event-handlers instead ('before:check', 'success:check')
        function preAuthCallback(o) {
            if (o && _.isFunction(o.preAuthCallback)) {
                o.preAuthCallback();
                return true;
            }
            return false;
        }
        // TODO: maybe it's possible to use api-event-handlers instead ('fail:check')
        function failAuthCallback(o) {
            if (o && _.isFunction(o.failAuthCallback)) {
                o.failAuthCallback();
            }
        }

        function callback(o, auth) {
            if (o && _.isFunction(o.callback)) {
                o.callback(auth);
            }
        }

        return function (o) {
            var preAuthDone;
            var type = (o && o.type) ? o.type : 'pgp';
            // stored auth code
            if (ox.guard.hasAuth(type)) {
                preAuthDone = preAuthCallback(o);
            }
            return authAPI.check(type).then(function (data) {
                data.forEach(function (d) {
                    storeAuth(d.auth, d.minutesValid, d.type);
                    // OK only if exists and still time left.
                    if (d.type && d.type.toLowerCase() === type.toLowerCase()) {
                        if (!preAuthDone) preAuthCallback(o);
                        callback(o, d.auth);
                    }
                });
                return data;
            }, function (e) {
                if (ox.debug) console.log(e);
                if (preAuthDone) failAuthCallback(o);
                return $.Deferred().reject('fail');
            });
        };
    })();

    // Authenticate against the Guard server, checking for specific key ID, return settings info
    // TODO: remove unused params userid/extra (check other potentially calling modules like office first)
    function authorize(userid, password, type, keyid) {
        return util.getUsersPrimaryEmail().then(function (email) {
            if (!email) return $.Deferred().reject();
            return loginAPI.login({
                encr_password: password,
                lang: ox.language,
                email: email,
                type: type
            }, keyid, type);
        });
    }

    function hasSmime() {
        return capabilities.has('smime');
    }

    function hasSmimeEnabled() {
        return hasSmime() && guardSettings.get('smime');
    }

    function hasGuardMail() {
        return capabilities.has('guard') && capabilities.has('guard-mail');
    }

    function hasGuard() {
        return capabilities.has('guard');
    }

    function hasPGPandSmime() {
        return hasSmimeEnabled() && hasGuardMail();
    }

    function isGuest() {
        return capabilities.has('guest');
    }

    // Simple call to verify password.  Keyid optional
    function verifyPassword(password, keyid, type) {
        return authorize(undefined, password, type, keyid).then(function (data) {
            if (data.auth && data.auth.length > 20) {
                return; // Good auth
            }
            require(['oxguard/auth'], function (auth) {
                auth.handleFail(data.auth);
            });
            return $.Deferred.reject();
        });
    }


    return {
        auth: authorize,
        checkAuth: checkAuth,
        checkJsonOK: checkJsonOK,
        getPassword: getPassword,
        htmlSafe: htmlSafe,
        metrics: metrics,
        savePassword: savePassword,
        showError: showError,
        storeAuth: storeAuth,
        verifyPassword: verifyPassword,
        // moved to util
        autoCryptEnabled: util.autoCryptEnabled,
        sanitize: util.sanitize,
        validateEmail: util.validateEmail,
        hasSmime: hasSmime,
        hasSmimeEnabled: hasSmimeEnabled,
        hasGuardMail: hasGuardMail,
        hasGuard: hasGuard,
        hasPGPandSmime: hasPGPandSmime,
        isGuest: isGuest
    };
});

