/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

define('oxguard/mail_lock/toggle-encryption', [
    'io.ox/backbone/mini-views/dropdown',
    'io.ox/backbone/mini-views/helplink',
    'oxguard/auth',
    'gettext!oxguard',
    'settings!oxguard',
    'less!oxguard/mail_lock/style'
], function (Dropdown, HelpView, auth, gt, settings) {

    'use strict';

    var ToggleEncryptionView = Backbone.View.extend({
        tagName: 'a',
        className: 'toggle-encryption',
        initialize: function (options) {
            this.mailModel = options.mailModel;
            this.listenTo(this.model, 'change:encrypt', function (model, val) {
                this.$('i.fa').toggleClass('encrypted', val);
                //gt('Click to enable encryption')
                //gt('Click to disable encryption')
            });
        },
        events: {
            'click': 'toggle',
            'change:encrypt': 'changed'
        },
        toggle: function (e) {
            e.preventDefault();
            if (this.encryption_forced) {
                require(['io.ox/core/notifications'], function (notify) {
                    notify.yell('error', gt('Reply must be encrypted'));
                });
            } else {
                var model = this.model;
                var mailModel = this.mailModel;
                if (!model.get('encrypt')) {
                    auth.ensureSetup().then(function () {
                        model.set('encrypt', !model.get('encrypt'));
                        if (!model.get('encrypt')) {  // If removing encryption, check for pgp attachments
                            checkPGPAttachment(mailModel);
                        }
                    });
                } else {
                    model.set('encrypt', !model.get('encrypt'));
                }
            }
        },
        forceEncryption: function () {
            this.encryption_forced = true;
            this.model.set('encrypt', true);
            this.render();
        },
        render: function () {
            this.$el.attr('tabindex', '-1');
            var guard = ox.guard.hasGuardMail();
            if (guard || this.encryption_forced || ox.guard.hasSmime()) {
                if (!_.device('small')) {
                    this.$el.empty().append(
                        $('<i class="fa fa-2x" aria-hidden="true">')
                            .toggleClass('encrypted', !!this.model.get('encrypt'))
                    );
                    this.$el.attr('title', gt('Enable Encryption')).attr('href', '#').attr('role', 'button');
                }
            }
            this.listenTo(this.model, 'change:encrypt', function () {
                this.changed();
            });
            if (this.model.get('encrypt')) {
                var view = this.view;
                window.setTimeout(function () {
                    showAttLink(view, false);
                }, 100);
            }
            return this;
        },
        noLinkMail: function (view) {
            this.view = view;
            checkAttLinkView(this.mailModel, view);
        },
        changed: function () {
            this.$el.attr('title', this.model.get('encrypt') ? gt('Disable Encryption') : gt('Enable Encryption'));
            if (!ox.guard.hasGuardMail() && ox.guard.hasSmime()) {  // If we don't have guard-mail, but has smime, then default to smime
                this.model.set('smime', true);
            }
            if (this.model && this.model.get('encrypt') && !this.model.get('smime')) {
                showAttLink(this.view, false);
                checkAttLink(this.mailModel);
                try {
                    var fromArray = this.mailModel.get('from');
                    var from = _.isArray(fromArray[0]) ? fromArray[0][1] : fromArray[1];
                    if (fromArray && ox.guard.get('primaryEmail') !== undefined) {
                        if (ox.guard.get('primaryEmail') !== from) {
                            var warn = settings.get('warnEmailDifferent');
                            if (warn === undefined || warn === false) {
                                displayWarning();
                            }
                        }
                    }
                } catch (e) {
                    console.log(e);
                }
            } else {
                showAttLink(this.view, true);
            }
        }
    });
    /*
    // TODO-784: get rid of
    var ToggleEncryptionViewMobile = Backbone.View.extend({
        tagName: 'button',
        initialize: function (model, mailModel) {
            this.mailModel = mailModel;
            this.listenTo(this.model, 'change:encrypt', function (model, val) {
                this.$('i.fa').toggleClass('encrypted', val);
                //gt('Click to enable encryption')
                //gt('Click to disable encryption')
            });
        },
        className: 'toggle-encryption',
        events: {
            'click': 'toggle',
            'change:encrypt': 'changed'
        },
        toggle: function () {
            if (this.encryption_forced) {
                require(['io.ox/core/notifications'], function (notify) {
                    notify.yell('error', gt('Reply must be encrypted'));
                });
            } else {
                var model = this.model;
                if (!model.get('encrypt')) {
                    auth.ensureSetup().then(function () {
                        model.set('encrypt', !model.get('encrypt'));

    // TODO-784: get rid of
    var ToggleEncryptionViewMobile = Backbone.View.extend({
        tagName: 'button',
        initialize: function (model, mailModel) {
            this.mailModel = mailModel;
            this.listenTo(this.model, 'change:encrypt', function (model, val) {
                this.$('i.fa').toggleClass('encrypted', val);
                //gt('Click to enable encryption')
                //gt('Click to disable encryption')
            });
        },
        className: 'toggle-encryption',
        events: {
            'click': 'toggle',          checkPGPAttachment(this.mailModel);
                        }
                    });
                } else {
                    model.set('encrypt', !model.get('encrypt'));
                }
            }
        },
        forceEncryption: function () {
            this.encryption_forced = true;
            this.model.set('encrypt', true);
            this.render();
        },
        render: function () {
            this.$el.attr('tabindex', '-1');
            var guard = ox.guard.hasGuardMail();
            if (guard || this.encryption_forced) {
                if (!_.device('small')) {
                    this.$el.empty().append(
                        $('<i class="fa fa-2x" aria-hidden="true">')
                            .toggleClass('encrypted', !!this.model.get('encrypt'))
                    );
                    this.$el.attr('title', gt('Enable Encryption'));
                }
            }
            this.listenTo(this.model, 'change:encrypt', function () {
                this.changed();
            });
            if (this.model.get('encrypt')) {
                var view = this.view;
                window.setTimeout(function () {
                    showAttLink(view, false);
                }, 100);
            }
            return this;
        },
        noLinkMail: function (view) {
            this.view = view;
            checkAttLinkView(this.mailModel, view);
        },
        changed: function () {
            this.$el.attr('title', this.model.get('encrypt') ? gt('Disable Encryption') : gt('Enable Encryption'));
            if (this.model && this.model.get('encrypt') && !(this.model.get('smime'))) {
                showAttLink(this.view, false);
                checkAttLink(this.mailModel);
                try {
                    var fromArray = this.mailModel.get('from');
                    var from = _.isArray(fromArray[0]) ? fromArray[0][1] : fromArray[1];
                    if (fromArray && ox.guard.get('primaryEmail') !== undefined) {
                        if (ox.guard.get('primaryEmail') !== from) {
                            var warn = settings.get('warnEmailDifferent');
                            if (warn === undefined || warn === false) {
                                displayWarning();
                            }
                        }
                    }
                } catch (e) {
                    console.log(e);
                }
            } else {
                showAttLink(this.view, true);
            }
        }
    });
    */

    // Hide or show mail attachment link
    function showAttLink(view, show) {
        if (view) {
            view.$('.attachments').toggleClass('guard-encrypted', !show);
        }
    }

    // Check if Mail Attachment link is shown and encryption enabled
    function checkAttLinkView(model, view) {
        if (model.get('share_attachments')) {
            if (model.get('share_attachments').enable) {
                if (model.get('encrypt')) {
                    showAttLink(view, false);
                }
            }
        }
    }

    // Show a warning if Mail Attachment Link is shown and encryption clicked
    function checkAttLink(model) {
        if (model.get('sharedAttachments')) {
            if (model.get('sharedAttachments').enabled) {
                require(['io.ox/core/tk/dialogs', 'settings!io.ox/mail'], function (dialogs, mail) {
                    var dialog = new dialogs.CreateDialog({ width: 450, height: 300, center: true, enter: 'ok' });
                    dialog.header($('<h4>').text(gt('Not Supported')));
                    var text = $('<p>').text(gt('%s is not supported with secured email and will be disabled.', mail.get('compose/shareAttachments/name')));
                    dialog.getBody().append(text);
                    dialog
                    .addPrimaryButton('ok', gt('OK'), 'ok')
                    .on('ok', function () {
                        $('.share-attachments').find(':checkbox').prop('checked', false);
                        var share = model.get('sharedAttachments');
                        share.enabled = false;
                        model.set('sharedAttachments', share);
                    })
                    .show();
                });
            }
        }
    }

    // Warning that the from address is different from primary
    function displayWarning() {
        require(['io.ox/core/tk/dialogs'], function (dialogs) {
            var dialog = new dialogs.CreateDialog({ width: 450, height: 300, center: true, enter: 'ok' });
            dialog.header($('<h4>').text(gt('Warning')));
            var text = $('<p>').text(gt('Sending an encrypted email from an account other than %s may cause problems with the recipient being able to reply.  Consider using your primary account.', ox.guard.get('primaryEmail')));
            var warn = $('<input type="checkbox" id="warnEmail">');
            var warntext = $('<span class="selectable" style="padding-left:10px;">').text(gt('Do not warn me again'));
            warntext.click(function () {
                warn.click();
            });
            warn.click(function (e) {
                settings.set('warnEmailDifferent', $(e.target).is(':checked'));
                settings.save();
            });
            dialog.getBody().append(text).append('<hr>').append(warn).append(warntext);
            dialog
            .addPrimaryButton('ok', gt('OK'), 'ok')
            .show();
        });
    }

    // Check to see if there are pgp attachments.  Checked if encryption removed
    // Likely the recipient won't be able to open this attachment if not sent through Guard
    function checkPGPAttachment(model) {
        var attachments = model.get('attachments');
        if (attachments && _.isArray(attachments.models)) {
            attachments.models.forEach(function (attModel) {
                if (attModel.get('file-mimetype') === 'application/pgp-encrypted' ||
                    (attModel.get('filename') && attModel.get('filename').indexOf('.pgp') > 0)) {
                    require(['io.ox/core/yell'], function (yell) {
                        //#.  Warning to the user after they removed encryption from an email.  One of the attachments is a pgp item, and it will likely be unreadable to the recipient if they remove encryption
                        yell('warning', gt('Sending encrypted attachments without encrypting the email will usually result in the attachment being unreadable by the recipient.'));
                    });
                }
            });
        }
    }

    return {
        View: ToggleEncryptionView
    };
});
