/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

define('oxguard/mail/security_model', ['gettext!oxguard'], function (gt) {

    'use strict';

    var SecurityModel = Backbone.Model.extend({
        initialize: function (init, mailModel) {
            this.mailModel = mailModel;
            this.set('sign', init.sign);
            this.set('encrypt', init.encrypt);
            this.set('pgpInline', init.pgpInline);
            this.set('authToken', init.authToken);
            this.set('smime', init.smime === true || init.type === 'smime');
            this.listenTo(this, 'change', this.onChange);

            if (!ox.guard.hasGuardMail() && ox.guard.hasSmime()) this.set('smime', true);  // No pgp mail avail
        },
        setMailModel: function (model) {
            this.mailModel = model;
        },
        events: {
            'change': 'onChange',
            'set': 'onChange'
        },
        defaults: {
            encrypt: false,
            sign: false
        },
        toJson: function () {
            return _.extend(
                {}, (this.mailModel ? this.mailModel.get('security') : {}), {
                    encrypt: this.get('encrypt'),
                    sign: this.get('sign'),
                    pgpInline: this.get('pgpInline'),
                    authToken: this.get('authToken'),
                    type: this.get('smime') ? 'smime' : 'pgp'
                });
        },
        onChange: function (model) {
            if (this.mailModel) {
                // Encrypting
                if (this.get('encrypt')) {
                    // For PGP encryption, we don't updateMailModel until auth is done
                    if (!this.get('authToken') && !this.get('smime')) {
                        this.set('encrypt', false);
                        this.checkAuth(model);
                    } else {
                        this.updateMailModel(model);
                    }
                    if (this.get('smime')) {
                        checkSmimeKeys(model);
                    }
                } else {
                    this.updateMailModel(model);
                }
                // Signing
                if (this.get('sign') && this.get('smime')) {
                    checkSmimeKeys(model);
                }
            }
        },
        updateMailModel: function (model) {
            var mailModel = this.mailModel;
            var savit = _.debounce(function () {
                mailModel.set('security', model.toJson());
                mailModel.save();
            }, 500);
            if (mailModel.requestSave) {
                savit();
            } else {
                window.setTimeout(function () {
                    savit();
                }, 1000);
            }
        },
        checkAuth: function (model) {
            if (this.get('tour')) {
                return;
            }
            var that = this;
            // Authentication required for pgp
            if (!model.get('smime')) {
                require(['oxguard/auth'], function (auth) {
                    var o = {
                        type: 'pgp'
                    };
                    auth.authorize(undefined, o).then(function (authcode) {
                        model.set('authToken', authcode);
                        model.set('encrypt', true);
                        that.updateMailModel(model);
                    }, function () {
                        model.set('encrypt', false);
                        that.updateMailModel(model);
                    });
                });
            } else {
                this.updateMailModel(model);
            }
        },
        setInline: function () {
            this.set('pgpInline', true);
        },
        setMime: function () {
            this.set('pgpInline', false);
        }
    });

    // Check that we have the required smime keys for signature
    function checkSmimeKeys(model) {
        var from = model.mailModel.get('from');
        var fromEmail = from[1];
        function check() {
            if (!ox.guard.hasSmimeKeys()) {
                require(['io.ox/core/yell'], function (yell) {
                    yell('error', gt('No S/MIME keys available.  Please add keys through the settings page.'));
                });
                model.set('sign', false);
                model.set('encrypt', false);
                return;
            }
            var match = false;
            ox.guard.get('smimeKeyList').forEach(function (k) {
                if (k.email.toLowerCase() === fromEmail.toLowerCase()) {
                    match = true;
                }
            });
            if (!match) {
                require(['io.ox/core/yell'], function (yell) {
                    yell(
                        'error',
                        gt('No Smime keys for this email address available for signing.  Please add the key through the settings page.'));
                });
                model.set('sign', false);
            }
        }
        if (ox.guard.hasSmimeKeys() === undefined) {
            require(['oxguard/api/certs'], function (keys) {
                keys.getCertificates().then(function () {
                    check();
                });
            });
        } else check();
    }

    return SecurityModel;
});
