/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

define('oxguard/mail/options_new', [
    'oxguard/mail/oxguard_mail_compose_core',
    'oxguard/mail/util',
    'oxguard/util',
    'settings!oxguard',
    'gettext!oxguard',
    'less!oxguard/mail/style'
], function (core, util, coreUtil, settings, gt) {

    'use strict';

    var lockclass = 'icon-lock icon-2x fa fa-lock fa-2x oxguard_lock';

    function createlock(baton) {
        baton.view.listenTo(baton.view.securityModel, 'change:encrypt', function () {
            setLock(baton);
        });
        setLock(baton);
    }

    function setLock(baton) {
        if (baton.view.securityModel.get('encrypt')) {
            core.lock(baton);
        } else {
            core.unlock(baton);
        }
    }

    function warnHTML(baton) {
        if (baton.view.securityModel.get('smime')) return;  // Doesn't apply to smime
        var inline = baton.view.securityModel.get('pgpInline');
        var encrypt = baton.view.securityModel.get('encrypt');
        if (!inline || !encrypt) return;  // Don't display warning if not Inline or not encrypted
        if (baton.config.get('editorMode') === 'text') return;
        if (baton.view.securityModel.get('warned')) return;  // Only warn once
        baton.view.securityModel.set('warned', true);
        require(['io.ox/core/tk/dialogs'], function (dialogs) {
            var dialog = new dialogs.CreateDialog({ width: 450, height: 300, center: true });
            dialog.header($('<h4>').text(gt('HTML Formatted Email')));
            var body = $('<p>');
            body.append(gt('HTML formatted emails may cause issues with PGP inline.  Click OK to change to Plain Text formatted Emails.'));
            dialog.getBody().append(body);
            dialog
            .addPrimaryButton('ok', gt('OK'), 'okx')
            .addButton('cancel', gt('Cancel'), 'cancel')
            .on('ok', function () {
                baton.config.set('editorMode', 'text');
                baton.config.on('change:editorMode', function () {
                    if ((baton.config.get('editorMode') === 'html') && (baton.view.securityModel.get('pgpInline'))) {
                        var dialog2 = new dialogs.CreateDialog({ width: 450, height: 300, center: true });
                        dialog2.header($('<h4>').text(gt('HTML Formatted Email')));
                        var warn = $('<p>').append(gt('HTML format is not recommended when using PGP Inline'));
                        dialog2.getBody().append(warn);
                        dialog2
                        .addPrimaryButton('ok', gt('OK'), 'okx')
                        .show();
                    }
                });
            })
            .show();
        });
    }

    function getType(current) {
        if (current === null || current === undefined) {  // if no decoded data, then return default
            if (settings.get('defaultInline')) {
                return ('inline');
            } return ('mime');
        }
        if (current.pgpFormat === 'pgpmime') return ('mime');
        return ('inline');
    }

    function createOptions(baton, count) {
        if (!ox.guard.isLoaded()) {
            coreUtil.addOnLoaded(function () {
                createOptions(baton, count);
            });
            return;
        }
        if (window.oxguarddata.action === 'sendfile') { // If it is going to be sending already encrypted file, then don't add encryption options
            window.oxguarddata.action = null;
            return;
        }

        var securityModel = baton.view.securityModel;

        // If this is a secure reply, then disable all of the options to make it plain text
        if ((window.oxguarddata.action === 'secureReply') || (window.oxguarddata.action === 'fileAttach') || (util.hasEncryptedAttachment(baton) > 0)) {
            if (window.oxguarddata.replyInline === true) {
                securityModel.setInline();
                window.oxguarddata.replyInline = null;
            }
            if (securityModel.get('pgpInline') === undefined) { // If not previously defined
                securityModel.set('pgpInline', getType(window.oxguarddata.current) === 'inline');
            }
            securityModel.set('encrypt', true);
            if (ox.guard.getSettings().secureReply === true) {
                baton.view.$el.find('#statuslock')
                    .attr('class', lockclass)
                    .click(false)
                    .attr('title', gt('Reply must be encrypted'));
                baton.view.$el.find('[data-name="encrypt"]').click(false);
            }
            window.oxguarddata.action = null;
        }

        var headers = baton.model.get('headers');
        if (headers && (headers['X-OxGuard-Sign'] === 'true')) {
            securityModel.set('sign', true);
        }

        createlock(baton);

        // Guard bug GUARD-200, deleted user still has defaults applied.  Verify configured before applying
        if (baton.config.get('newmail') === true && ((coreUtil.hasGuardMailCapability() && coreUtil.isGuardConfigured()) || coreUtil.hasSmime())) {  // New email doesn't have recips yet.  Check to see if there are defaults
            if (settings.get('defaultSmime') && settings.get('smime') && coreUtil.hasSmime()) {
                securityModel.set('smime', true);
            }
            if (settings.get('defaultInline')) {
                securityModel.setInline();
            } else securityModel.setMime();

            baton.config.unset('newmail');

            if (settings.get('defaultSign')) {
                securityModel.set('sign', true);
            }
        }

        if (baton.config.get('type') !== 'new' && baton.config.get('type') !== 'edit') {  // set for replies, forward
            if (coreUtil.hasSmime() || coreUtil.hasGuardMailCapability()) {
                var original = _.isArray(baton.model.original) ? baton.model.original[0] : baton.model.original;
                var origSmime = original && original.security && (original.security.type === 'SMIME');
                if ((settings.get('defaultSmime') || origSmime) && settings.get('smime') && coreUtil.hasSmime()) {
                    securityModel.set('smime', true);
                }
                if (settings.get('defaultSign')) {
                    securityModel.set('sign', true);
                }
            }
        }

        securityModel.on('change:pgpInline change:encrypt', function () {
            warnHTML(baton);
        });

        // If default is to use guard, go ahead and lock
        if (settings.get('defaultEncrypted')) {
            securityModel.set('encrypt', true);
            core.lock(baton);
        }

        securityModel.on('change:pgpInline', function () {
            securityModel.set('encrypt', true);
        });

        securityModel.on('change:PubKey', function () {
            if (securityModel.get('PubKey')) {
                attachPubKey(baton);
            }
        });

    }
    // Attach the users public key to the email
    function attachPubKey(baton) {
        var attachments = baton.model.get('attachments');
        var securityModel = baton.view.securityModel;
        for (var i = 0; i < attachments.models.length; i++) {
            if (attachments.models[i].get('guard') === 'pubkey') {
                securityModel.set('PubKey', false);
                return;
            }
        }
        require(['oxguard/api/keys', 'io.ox/core/notifications', 'io.ox/mail/compose/api'], function (keysAPI, notify, composeApi) {
            keysAPI.download({ keyType: 'public' })
            .done(function (data) {
                try {
                    var file = new Blob([data.key], { type: 'application/pgp-keys' });
                    file.name = 'public.asc';
                    composeApi.space.attachments.add(baton.model.get('id'), { file: file })
                    .then(function (data) {
                        baton.model.attachFiles(data);
                    });
                } catch (e) {
                    console.error(e);
                    notify.yell('error', gt('Unable to attach your public key'));
                }

            })
            .fail(function () {
                notify.yell('error', gt('Unable to attach your public key'));
            })
            .always(function () {
                securityModel.set('PubKey', false);
            });
        });
    }

    return {
        createOptions: createOptions,
        warnHTML: warnHTML
    };
});
