/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

define('oxguard/mail/mailReplyActions', [
    'settings!oxguard',
    'io.ox/core/extensions'
], function (settings, ext) {

    'use strict';

    ext.point('io.ox/mail/compose/boot').extend({
        id: 'detectNewMail',
        index: 'first',
        perform: function (baton) {
            // returns here if any other type than new (e.g. replies etc)
            if (baton.data && baton.data.type && baton.data.type.toLowerCase() !== 'new') return;
            // returns here, if restorepoint or copy of draft is opened
            if (baton.data && baton.data.id) return;
            baton.config = _.extend({}, baton.config, { newmail: true });
        }
    });

    function checkRestricted(baton) {
        if (baton.model.get('security') && baton.view.securityModel) {
            if (baton.model.get('security').encrypt && !baton.view.securityModel.get('smime')) {  // if pgp reply, check if forced encryption
                require(['oxguard/util'], function (util) {
                    if (!util.hasGuardMailCapability() || util.isGuest()) {
                        baton.view.toggleEncryption.forceEncryption();
                        disableRecipients(baton.view);
                    }
                });
                //baton.model.save();
            }
        }
    }

    ext.point('io.ox/mail/compose/boot').extend({
        id: 'checkSecuritySettings',
        index: 1001,
        perform: function (baton) {
            if (baton.data.original) {
                var original = baton.data.original;
                if (_.isArray(baton.data.original) && baton.data.original.length > 0) {
                    original = baton.data.original[0];
                }
                if (original.security && original.security.decrypted) {
                    baton.view.securityModel.set('smime', original.security.type === 'SMIME');
                    baton.view.securityModel.set('authToken', original.security.authentication, true);
                    baton.view.securityModel.set('encrypt', true);
                }
            }
            if (baton.data.type === 'reply' || baton.data.type === 'replyall' || baton.data.type === 'forward') {
                checkReply(baton);
                checkRestricted(baton);
                return;
            }
            if (baton.data.type === 'edit') {
                checkEdit(baton);
                checkRestricted(baton);
                return;
            }

            if (baton.config && baton.config.newmail) return;   // New mail, no previous settings

            if (baton.model.get('security')) {
                baton.view.securityModel.set('smime', baton.model.get('security').type === 'smime');
                baton.view.securityModel.set('sign', baton.model.get('security').sign, { silent: true });
                checkRestricted(baton);
            }
        }
    });

    function getOriginal(baton) {
        if (_.isArray(baton.data.original)) {
            return baton.data.original[0];
        }
        return baton.data.original;
    }

    // If loading a draft email, need to set security settings same as before
    function checkEdit(baton) {
        var obj = getOriginal(baton);
        if (obj.headers && obj.headers['X-Security']) {
            var reply = obj.headers['In-Reply-To'] !== undefined;  // If this is editing a reply
            var securitySettings = _.isArray(obj.headers['X-Security']) ? obj.headers['X-Security'][0] : obj.headers['X-Security'];
            var settings = securitySettings.split(';');
            settings.forEach(function (setting) {
                var s = setting.split('=');
                var value = (s[1] ? (s[1].indexOf('true') > -1) : false);
                switch (s[0].trim()) {
                    case 'sign':
                        baton.view.securityModel.set('sign', value);
                        break;
                    case 'encrypt':
                        baton.view.securityModel.set('encrypt', value);
                        if (value === true && reply) {  // Check if we should enforce the secure reply here
                            if (ox.guard.getSettings().secureReply) {
                                baton.view.toggleEncryption.forceEncryption();
                            }
                        }
                        break;
                    case 'pgpInline':
                        if (baton.view.securityModel.get('encrypt')) baton.view.securityModel.set('pgpInline', value);
                        break;
                    default:
                        console.error('Unknown security parameter ' + s[0]);
                }
            });
            if (obj.security.authentication) {
                baton.view.securityModel.set('authentication', obj.security.authentication);
            }
            baton.model.unset('security_info');
        }
    }

    // When doing reply/forward, check essential security data moved to new model
    function checkReply(baton) {
        var obj = getOriginal(baton);
        var securityModel = baton.view.securityModel;
        if (obj.security && obj.security.decrypted) {
            securityModel.set('encrypt', true);
            var security = baton.model.get('security');
            security.encrypt = true;
            if (obj.security.authentication) {
                security.authentication = obj.security.authentication;
            }
            var headers = baton.model.get('headers') || {};
            baton.model.set('headers', _.extend(headers, { 'X-Guard-msg-ref': obj.id }));
            security.msgRef = obj.id;
            if (ox.guard.getSettings().secureReply) {
                baton.view.toggleEncryption.forceEncryption();
            }
            baton.model.set('security', _.extend(baton.model.get('security'), { authentication: obj.security.authentication }));
        }
        baton.model.unset('security_info');

    }

    function disableRecipients(view) {
        view.$el.find('.recipient-actions').remove();
        view.$el.find('.tokenfield').css('background-color', '#e0e0e0');
        view.$el.find('.token-input').click(function (e) {
            $(e.currentTarget).blur();
            e.preventDefault();
            require(['gettext!oxguard', 'io.ox/core/notifications'], function (gt, notify) {
                notify.yell('error', gt('Recipients cannot be added to this encrypted reply'));
            });
        });
        view.$el.find('.twitter-typeahead').hide();
    }

});
