/*
 *
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

define('oxguard/mail/checker', [
    'io.ox/core/extensions',
    'io.ox/core/extPatterns/links',
    'oxguard/core/og_http',
    'oxguard/oxguard_core',
    'io.ox/core/capabilities',
    'gettext!oxguard'
], function (ext, links, http, core, capabilities, gt) {

    'use strict';

    var firstload = true;

    function upsell(baton) {
        var loc = baton.view.$el;
        if (loc.find('.guardUpsell').length < 1) { // Make sure not already drawn
            require(['oxguard/mail/upsell'], function (upseller) {
                upseller.upsell(loc);
            });
        }
    }

    // Do the check if the email is PGP email.  Check for inline, MIME, or signature
    function checkPGP(baton, location) {
        if (baton.data.security && baton.data.security.decrypted) {
            if (!ox.guard.hasGuardMail() && !capabilities.has('guest') && baton.data.security.type === 'PGP') { // If not guard user, display upsell if any
                upsell(baton);
            }
            return; // If already decoded
        }
        if (baton.view.model === null) {
            return;
        }
        var headers = baton.view.model.get('headers');
        if (headers === undefined) { // not yet loaded
            return;
        }
        // Init
        if (baton.data.headers === undefined) { // If email wasn't fully loaded into baton, do so now.
            baton.data = baton.view.model.toJSON();
        }
        var mail = baton.data;
        if (mail.results !== undefined) return;

        if (headers['X-OXGUARD-GUEST'] !== undefined) {
            baton.view.model.set('pgp_found', true);
            pgpAttach(baton);
            pgpFound.call(this, baton);
        }
        // Main check if Guard email
        if (baton.view.model.get('security') && baton.view.model.get('security').decrypted) return;  // Bug 53622, loop in mobile view
        if (mail.security_info && checkCapabilities(mail.security_info)) {
            if (mail.security_info.encrypted) {
                baton.view.model.set('pgp_found', true);
                pgpAttach(baton);
                pgpFound.call(this, baton, location);
            } else if (mail.security_info.signed) {
                if (!mail.security && !baton.signatureCheck) pullAgain(false, baton);  // If we don't already have result, reload
                baton.signatureCheck = true;  // avoid loops
            }
        } else if (mail.security_info && mail.security_info.encrypted) {
            baton.data.attachments.forEach(function (a) {
                if (a.disp === 'inline') {
                    a.content = gt('Encrypted email');
                }
            });
            baton.data.unsupported = true;
            upsell(baton);
        }

    }

    function checkCapabilities(security) {
        if (security.type === 'SMIME') {
            if (capabilities.has('smime')) return true;
            return false;
        }
        return capabilities.has('guard');
    }

    function showError(data, baton) {
        var type = (baton.data.security_info && baton.data.security_info.type) ? baton.data.security_info.type : 'pgp';
        require(['oxguard/oxguard_core', 'oxguard/api/auth'], function (core, authAPI) {
            authAPI.resetToken(type);
            core.showError(data);
            if (!baton.signatureCheck) baton.view.redraw();  // avoid loop
        });
    }

    function pullAgain(decrypt, baton) {
        require(['io.ox/mail/api'], function (api) {
            var options = {
                cache: false
            };
            var obj = {
                id: baton.data.id,
                folder_id: baton.data.folder_id
            };
            if (decrypt) {
                obj.decrypt = 'true';
            } else {
                obj.verify = 'true';
            }
            api.get(obj, options)
            .then(function (data) {
                if (!data.security) {
                    console.error('Problem getting security response from guard');
                    return;
                }
                data = removeSignatureFile(data); // remove verified signatures
                if (baton.sessionKey) { // If local decrypt, save the sessionKey for attachments, etc
                    data.security.sessionKey = baton.sessionKey;
                }
                var origData = _.extend(baton.data, {});
                baton.model.set(data);
                origData.view = baton.view;
                baton.model.set('origMail', origData);
                // Update the pool mail with the attachment and security data of the email.  Otherwise will
                // be overwritten with changes in listViews, sort order, etc
                var stored = api.pool.get('detail').get(_.cid(data));
                if (stored) {
                    stored.set('guardIcons', _.pick(data, 'attachment', 'security'));
                }
                delete (baton.view.attachmentView); // Need to redo the attachment list
                baton.view.redraw();
            }, function (data) {
                showError(data, baton);
            });
        });
    }

    // Remove signature.asc files from attachments if signatures verified
    function removeSignatureFile(data) {
        if (data.security && data.security.signatures) {
            var verified = true;
            data.security.signatures.forEach(function (d) {
                verified = verified && d.verified;
            });
            if (verified) {
                var newAttachments = [];
                data.attachments.forEach(function (att) {
                    if (att.content_type !== 'application/pgp-signature') {
                        newAttachments.push(att);
                    }
                });
                data.attachments = newAttachments;
            }
        }
        return data;
    }

    function busy(location) {
        $(location).busy();
    }

    // PGP found, do send for decode
    function pgpFound(baton, location) {
        var smime = (baton.data.security_info && baton.data.security_info.type === 'SMIME');
        var goFunction = function () {
            pullAgain(true, baton);
        };
        checkLoaded().then(function () {
            checkSession(smime).then(function (auth) {
                require(['oxguard/mail/oxguard_mail_password'], function (oxmail) {
                    if (auth && auth.length > 10) {  // If auth code in session, load decrypted
                        pullAgain(true, baton);
                    } else if (!smime && ox.guard.notReady()) {  // If we have auth, or needs further action
                        busy(location, oxmail.getId(baton));
                        checkauth(goFunction, baton, undefined);
                    } else {
                        // Otherwise, draw password prompt
                        $(location).replaceWith(oxmail.passwordPrompt(baton, false, location, goFunction));
                        baton.view.$el.find('.attachments').hide();
                        if (!_.device('ios')) {
                            baton.view.$el.find('#oxgrpass').focus();
                        }
                        baton.view.$el.show();
                    }
                });
            });
        });
    }

    function checkSession(smime) {
        var def = $.Deferred();
        var type = smime ? 'SMIME' : 'PGP';
        if (!firstload) {
            def.resolve(ox.guard.getAuth(type));
            return def;
        }
        require(['oxguard/oxguard_core'], function (core) {
            var o = {
                type: type
            };
            core.checkAuth(o)
            .done(function (data) {
                firstload = false;
                data.forEach(function (d) {
                    if (d.type === type) {
                        def.resolve(d.auth);
                    }
                });
                def.resolve();
            })
            .fail(function () {
                firstload = false;
                def.resolve();
            });
        });
        return def;
    }

    // Timeout loop waiting for the Guard backend data to be loaded
    function waitForLoad(def) {
        if (ox.guard.isLoaded()) {
            def.resolve();
            return;
        }
        setTimeout(function () {
            waitForLoad(def);
        }, 500);
    }

    // Check if Guard backend data loaded.  If not, start wait loop
    function checkLoaded() {
        var def = $.Deferred();
        if (ox.guard.isLoaded()) {
            def.resolve();
        } else {
            waitForLoad(def);
        }
        return def;
    }

    function checkauth(go, baton, index) {
        if (ox.guard.needsPassword()) { // If no password for user has been defined, open dialog
            require(['oxguard/core/tempPassword'], function (core) {
                core.createOxGuardPasswordPrompt(baton, go, '', undefined, index, location);
            });

            return;
        }
        if (ox.guard.needsKey()) { // If received a first encrypted email while already logged in, will still have 'no key' in passcode
            require(['oxguard/core/tempPassword', 'oxguard/oxguard_core'], function (password, core) {
                core.auth(ox.user_id, '') // Try to reload authorization to see if keys now exist
                .done(function (data) {
                    ox.guard.loadData(data);
                    if (data.auth === 'No Key') {
                        baton.model.unset('security_info');
                        baton.view.redraw();
                        return;  // Still no key, non guard generated email?
                    }
                    password.createOxGuardPasswordPrompt(baton, go, '', undefined, index, location);  // Create new password
                });

            });
            return;
        }
        go(baton, '', index, ox.guard.getAuth());
    }

    function pgpAttach(baton) {
        baton.data.PGP = true;
    }

    return {
        checkPGP: checkPGP,
        pullAgain: pullAgain,
        showError: showError,
        pgpFound: pgpFound
    };
});
