/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.storage.file.osgi;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.database.AssignmentFactory;
import com.openexchange.filestore.DatabaseAccessProvider;
import com.openexchange.filestore.FileStorageService;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.database.GuardDatabaseService;
import com.openexchange.guard.storage.Storage;
import com.openexchange.guard.storage.cache.FileCacheStorage;
import com.openexchange.guard.storage.file.internal.FileStorageImpl;
import com.openexchange.guard.storage.file.internal.LegacyFileStorageImpl;
import com.openexchange.guard.storage.file.internal.SproxydDatabaseProvider;
import com.openexchange.guard.storage.file.internal.SproxydStorageImpl;
import com.openexchange.osgi.HousekeepingActivator;

/**
 * {@link GuardFileStorageActivator}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class GuardFileStorageActivator extends HousekeepingActivator {

    private static final String SPROXYD_FILESTORE = "com.openexchange.guard.storage.file.sproxydFilestore";

    /**
     * Initialises a new {@link GuardFileStorageActivator}.
     */
    public GuardFileStorageActivator() {
        super();
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.osgi.DeferredActivator#getNeededServices()
     */
    @Override
    protected Class<?>[] getNeededServices() {
        return new Class<?>[] { GuardConfigurationService.class, FileCacheStorage.class,
            FileStorageService.class };
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.osgi.DeferredActivator#startBundle()
     */
    @Override
    protected void startBundle() throws Exception {
        final Logger logger = LoggerFactory.getLogger(GuardFileStorageActivator.class);
        logger.info("Starting bundle {}", context.getBundle().getSymbolicName());
        Services.setServiceLookup(this);

        GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
        if (guardConfigService.getProperty(GuardProperty.fileStorageType).isEmpty()) {
            registerService(Storage.class, new LegacyFileStorageImpl(guardConfigService));
        } else {
            switch (guardConfigService.getProperty(GuardProperty.fileStorageType)) {
                case "sproxyd":
                    String fileStore = guardConfigService.getProperty(SPROXYD_FILESTORE);
                    if (fileStore == null || fileStore.isEmpty()) {
                        fileStore = "guard";
                    }
                    registerService(Storage.class, new SproxydStorageImpl(this.getService(FileStorageService.class), fileStore));
                    trackService(GuardDatabaseService.class);
                    trackService(AssignmentFactory.class);
                    registerService(DatabaseAccessProvider.class, new SproxydDatabaseProvider(this));
                    break;
                default:
                    registerService(Storage.class, new FileStorageImpl(this.getService(FileStorageService.class), guardConfigService.getProperty(GuardProperty.uploadDirectory)));
            }
        }

        logger.info("GuardFileStorageService registered.");
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.osgi.HousekeepingActivator#stopBundle()
     */
    @Override
    protected void stopBundle() throws Exception {
        final Logger logger = LoggerFactory.getLogger(GuardFileStorageActivator.class);
        logger.info("Stopping bundle {}", context.getBundle().getSymbolicName());

        unregisterService(Storage.class);
        logger.info("GuardFileStorageService unregistered.");

        Services.setServiceLookup(null);
        super.stopBundle();
    }
}
