/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.servlets.pgpmail;

import java.util.HashMap;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import com.openexchange.exception.OXException;
import com.openexchange.guard.servlets.AbstractGuardServlet;
import com.openexchange.guard.servlets.GuardErrorResponseRenderer;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.BasicAuthServletAuthenticationHandler;
import com.openexchange.guard.servlets.authentication.OXGuardSessionAuthenticationHandler;
import com.openexchange.server.ServiceLookup;

/**
 * Provides PGP mailing functionality
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class PGPMailServlet extends AbstractGuardServlet {

    private static final long serialVersionUID = -3402143663178340210L;
    private static final String GET_PUBLIC_KEY_FROM_ATTACHMENT_ACTION = "savepublicattach";

    private static final String POST_PROCESS_INCOMING_MIME = "process_mime";
    private static final String POST_VERIFY_MESSAGE = "verify";
    private static final String POST_DECRYPT_MIME_MESSAGE = "decrypt";
    private static final String ENCRYPT_MIME = "encrypt_mime";

    private transient final HashMap<String, GuardServletAction> getActions;
    private transient final HashMap<String, GuardServletAction> postActions;

    /**
     * Initializes a new {@link PGPMailServlet}.
     * 
     * @throws OXException
     */
    public PGPMailServlet(ServiceLookup services) throws OXException {

        OXGuardSessionAuthenticationHandler userSessionAuthenticationHandler = new OXGuardSessionAuthenticationHandler();
        BasicAuthServletAuthenticationHandler basicAuthAuthenticationHandler = createBasicAuthHandler();

        //GET
        getActions = new HashMap<String, GuardServletAction>();
        getActions.put(GET_PUBLIC_KEY_FROM_ATTACHMENT_ACTION, new GetPublicKeyFromAttachmentAction().setAuthenticationHandler(userSessionAuthenticationHandler));

        //POST
        postActions = new HashMap<String, GuardServletAction>();
        postActions.put(POST_PROCESS_INCOMING_MIME,
            new MimeEmailAction()
                .setAuthenticationHandler(basicAuthAuthenticationHandler)
                .setErrorResponseRenderer(new GuardErrorResponseRenderer(HttpServletResponse.SC_INTERNAL_SERVER_ERROR)));
        postActions.put(POST_DECRYPT_MIME_MESSAGE,
            new DecryptMimeEmailAction()
                .setAuthenticationHandler(basicAuthAuthenticationHandler)
                .setErrorResponseRenderer(new GuardErrorResponseRenderer(HttpServletResponse.SC_INTERNAL_SERVER_ERROR)));
        postActions.put(POST_VERIFY_MESSAGE,
            new VerifyMimeEmailAction()
                .setAuthenticationHandler(basicAuthAuthenticationHandler)
                .setErrorResponseRenderer(new GuardErrorResponseRenderer(HttpServletResponse.SC_INTERNAL_SERVER_ERROR)));
        postActions.put(ENCRYPT_MIME, new EncryptAction(services).setAuthenticationHandler(basicAuthAuthenticationHandler));
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.servlet.http.HttpServlet#doGet(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response) {
        doAction(request, response, getActions);
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.servlet.http.HttpServlet#doPost(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response) {
        doAction(request, response, postActions);
    }
}
