/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.storage;

import java.io.IOException;
import java.io.InputStream;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.io.IOUtils;
import com.openexchange.exception.OXException;
import com.openexchange.guard.database.GuardShardingService;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.inputvalidation.StringNotContainsInputValidator;
import com.openexchange.guard.storage.cache.FileCacheItem;
import com.openexchange.guard.storage.cache.FileCacheStorage;
import com.openexchange.guard.storage.osgi.Services;

/**
 * {@link AbstractLegacyStorage}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public abstract class AbstractLegacyStorage implements Storage {

    /**
     * A pattern which matches the default directory prefix used by {@link AbstractLegacyStorage} {
     */
    protected Pattern prefixPattern = Pattern.compile("ext_(\\d*)_(\\d*)");

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.storage.Storage#readObj(int, int, java.lang.String)
     */
    @Override
    public InputStream readObj(String objectId) throws OXException {
        filterObjId(objectId);
        // First try to pull from cache
        FileCacheStorage storage = Services.getService(FileCacheStorage.class);
        FileCacheItem item = storage.getById(objectId);
        if (item != null && objectExists(item.getLocation())) {
            return getObjectStream(item.getLocation());
        }
        return null;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.storage.Storage#saveEncrObj(int, int, java.lang.String, byte[])
     */
    @Override
    public Boolean saveEncrObj(int userId, int contextId, String objectId, byte[] data) throws OXException {
        filterObjId(objectId);
        GuardShardingService sharding = Services.getService(GuardShardingService.class);
        String directoryPrefix = getDirectoryPrefix(userId, contextId, sharding.getShard(userId, contextId));

        String filename = writeObj(directoryPrefix, objectId, data);

        if (filename != null) {
            FileCacheStorage fileCacheStorage = Services.getService(FileCacheStorage.class);
            //Only inserting the file cache item if it does not exist yet
            //(It could exists when storing an attachment for multiple recipients)
            if (fileCacheStorage.getById(objectId) == null) {
                fileCacheStorage.insert(objectId, userId, filename);
            }
        }

        return filename != null;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.storage.Storage#copy(int, int, java.lang.String, java.lang.String)
     */
    @Override
    public boolean copy(int userId, int contextId, String sourceObjectId, String destinationObjectId) throws OXException {

        filterObjId(sourceObjectId);
        filterObjId(destinationObjectId);

        GuardShardingService sharding = Services.getService(GuardShardingService.class);
        String directoryPrefix = getDirectoryPrefix(userId, contextId, sharding.getShard(userId, contextId));

        //Copy
        try(InputStream objectStream = getObjectStream(directoryPrefix, sourceObjectId);){
            final String filename = writeObj(directoryPrefix, destinationObjectId, IOUtils.toByteArray(objectStream));
            if(filename != null) {
                FileCacheStorage fileCacheStorage = Services.getService(FileCacheStorage.class);
                //(It could exists when storing an attachment for multiple recipients)
                if (fileCacheStorage.getById(destinationObjectId) == null) {
                    fileCacheStorage.insert(destinationObjectId, userId, filename);
                }
            }
            return filename != null;
        } catch (IOException e) {
            throw GuardCoreExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.storage.Storage#getDirectoryPrefix(int, int, int)
     */
    @Override
    public String getDirectoryPrefix(int userId, int contextId, int shardId) throws OXException {
        String directory = "";
        if (contextId > 0) {
            directory = "ctx_" + contextId;
        } else {
            directory = "ext_" + shardId + "_" + userId;
        }
        return (directory);
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.storage.Storage#getGuestShardIdFromPrefix(java.lang.String)
     */
    @Override
    public Integer getGuestShardIdFromPrefix(String directoryPrefix) {
        directoryPrefix = Objects.requireNonNull(directoryPrefix, "directoryPrefix must not be null");
        Matcher matcher = prefixPattern.matcher(directoryPrefix);
        if(matcher.find()) {
           return Integer.parseInt(matcher.group(1));
        }
        return null;
    }

    /**
     * Filter for .. attack
     *
     * @param objectId
     * @throws OXException
     */
    private void filterObjId(String objectId) throws OXException {
        new StringNotContainsInputValidator("..").assertIsValid(objectId, "objectId");
    }
}
