/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.guest.impl.converters;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.Objects;
import java.util.Properties;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.internet.MimeMessage;
import com.openexchange.exception.OXException;
import com.openexchange.guard.guest.exceptions.GuardGuestExceptionCodes;
import com.openexchange.guard.guest.metadata.storage.GuardGuestEMailMetadataService;
import com.openexchange.guard.guest.metadata.storage.GuardGuestEmailMetadata;
import com.openexchange.guard.user.UserIdentity;

/**
 * {@link MetaDataConverter}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.0
 */
public class MetaDataConverter implements GuestItemConverter {

    private static final String                  DEFAULT_FOLDER        = "INBOX";
    private static final int                     DEFAULT_MESSAGE_FLAGS = 1 << 4; /** The recent flag. See {@link MailMessage} **/
    private final GuardGuestEMailMetadataService guardGuestEmailMetadataService;

    public MetaDataConverter(GuardGuestEMailMetadataService guardGuestEmailMetadataService) {
        this.guardGuestEmailMetadataService = Objects.requireNonNull(guardGuestEmailMetadataService, "GuardGuestEMailMetadataService service missing or null");
    }

    /**
     * Internal method to create a MimeMessage from the given stream
     *
     * @param inputStream The stream to create the MimeMessge from
     * @return The MimeMessage constructed from the given stream
     * @throws MessagingException
     */
    private MimeMessage createMimeMessage(InputStream inputStream) throws MessagingException {
        return new MimeMessage(Session.getDefaultInstance(new Properties()), inputStream);
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.guest.impl.converters.GuestItemConverter#convert(com.openexchange.guard.user.UserIdentity, java.io.InputStream, java.io.OutputStream, java.lang.String)
     */
    @Override
    public GuestItemConverterResult convert(UserIdentity guest, InputStream guardGuestData, OutputStream output, String objectId) throws OXException {
        // Check if meta data already exists
        if (guardGuestEmailMetadataService.get(guest.getOXGuardUser().getContextId(),
            guest.getOXGuardUser().getId(),
            objectId) != null) {
            return GuestItemConverterResult.NOT_CONVERTED_NOT_NECASSARY;
        }
        // Create
        MimeMessage msg;
        try {
            msg = createMimeMessage(guardGuestData);
            GuardGuestEmailMetadata metaData = guardGuestEmailMetadataService.createMetaDataFrom(objectId, DEFAULT_FOLDER, msg, DEFAULT_MESSAGE_FLAGS);
            guardGuestEmailMetadataService.insert(guest.getOXGuardUser().getContextId(), guest.getOXGuardUser().getId(), metaData);
        } catch (MessagingException e) {
            throw GuardGuestExceptionCodes.PROBLEM_WITH_CONVERSION.create(e);
        }
        return GuestItemConverterResult.NOT_CONVERTED_NOT_NECASSARY; // Nothing needed to be saved in output
    }

}
